/**
 *  Firmware Disclaimer Information
 *
 *  1. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *     code, which is supplied by BEST HEALTH ELECTRONIC Inc., (hereinafter referred to as BestHealth) is the
 *     proprietary and confidential intellectual property of BestHealth, and is protected by copyright law and
 *     other intellectual property laws.
 *
 *  2. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *     code, is confidential information belonging to BestHealth, and must not be disclosed to any third parties
 *     other than BestHealth and the customer.
 *
 *  3. The program technical documentation, including the code, is provided and for customer reference
 *     only. After delivery by BestHealth, the customer shall use the program technical documentation, including
 *     the code, at their own risk. BestHealth disclaims any expressed, implied or statutory warranties, including
 *     the warranties of merchantability, satisfactory quality and fitness for a particular purpose.
 *
 *  <h2><center>Copyright (C) BEST HEALTH ELECTRONIC Inc. All rights reserved</center></h2>
 */
#include <stdint.h>
#include "..\..\..\driver\wdt.h"
#include "..\..\..\driver\Ram.h"
#include "..\..\..\driver\Oscillators.h"
#include "..\..\..\driver\AfePower.h"
#include "..\..\..\driver\Adc24.h"
#include "..\..\..\driver\Uart.h"
#include "..\..\..\driver\TimeBase.h"
#include "..\..\..\application\BodyScales.h"

#define TIME_CYCLE_MS 500
#define TIMEOUT_MAX   (30000L / TIME_CYCLE_MS)
#define TIMEOUT_30S   (30000L / TIME_CYCLE_MS)
#define TIMEOUT_10S   (10000L / TIME_CYCLE_MS)
#define TIMEOUT_8S    (8000L / TIME_CYCLE_MS)

BodyScales_t scales;
uint16_t haltTimeMs;
uint8_t scalesStatusLast;

void WeightHardwareSetting(uint8_t sampling);
void SystemEnterHalt();
void SystemEnterWakeUp();

volatile uint8_t uartQueueBuf[20];    // 中断使用需要加 volatile

#define CAL0DATA_DEFAULT    (38555)    //!< 0點默認校準值  -33601
#define CAL1DATA_DEFAULT    (79830)    //!< 第1點默認校準值 60427
#define CAL2DATA_DEFAULT    (79900)    //!< 第2點默認校準值 91609
#define CAL3DATA_DEFAULT    (79850)    //!< 第3點默認校準值 91621
#define AUTOON_DATA_DEFAULT 8559       //!< 自動上稱對應的ADC值  79

void main()
{
    // 判斷是否為上電復位或者非正常情況下的復位, 如果是上電復位，執行上電復位初始化，反之執行WDT溢出初始化
    if (_to == 0 || _pdf == 0)
    {
        // config sys clock
        Ram_Clear();
        // config Oscillators
        Oscillators_Cfg();
        // config WDT
        Wdt_Enable(WDT_TIME_500MS);
        // timeBase
        TimeBase0_Cfg(TIMEBASE_500MS);
        TimeBase0_Isr_Enable();
        TimeBase0_Enable();
        // config uart0
        uart0Status.rxBuf    = uartQueueBuf;            // 傳入緩存Buf地址
        uart0Status.rxBufLen = sizeof(uartQueueBuf);    // 傳入緩存Buf長度
        Uart_Cfg_t uartCfg;
        uartCfg.flag.b.addr   = false;
        uartCfg.flag.b.wakeUp = true;
        Uart0_Cfg(&uartCfg);
        Uart0_Enable();
        Uart0_Isr_Enable();
        // enable emi
        _emi = 1;

        scales.flag.b.isNeedTare = true;    //上電去皮，更新零點
        BodyScales_EnterWeighting(&scales, true);
        WeightHardwareSetting(ADC24_RATE_10HZ);
        if (scales.calStatus == BODY_SCALES_CAL_STATUS_FAIL)
        {
            scales.adcCal0    = CAL0DATA_DEFAULT;
            scales.adcCal1    = CAL1DATA_DEFAULT;
            scales.adcCal2    = CAL2DATA_DEFAULT;
            scales.adcCal3    = CAL3DATA_DEFAULT;
            scales.autoOnData = AUTOON_DATA_DEFAULT;
        }

        haltTimeMs = TIMEOUT_MAX - TIMEOUT_30S;
    }
    else
    {
        // WDT溢出復位初始化
        GCC_CLRWDT();
        BodyScales_EnterAutoOn(&scales);
        WeightHardwareSetting(ADC24_RATE_1302HZ);
    }
    while (1)
    {
        GCC_CLRWDT();
        if (haltTimeMs >= TIMEOUT_MAX)
        {
            uint8_t txChar[] = "EnterHalt\n";
            Uart0_TxByte_Polling(txChar, sizeof(txChar));
            SystemEnterHalt();
        }
        Adc24_Poling();
        if (adc24SourceData.flag.b.isReady)
        {
            adc24SourceData.flag.b.isReady = 0;
            BodyScales_Loop(&adc24SourceData.data.adcData, &scales);
            if (scales.flag.b.isNeedTare)
            {
                uint8_t txChar[] = "Tare\n";
                Uart0_TxByte_Polling(txChar, sizeof(txChar));
            }
            else
            {
                if (scales.status == BODY_SCALES_STATE_NOLOAD)
                {
                    // 空載/當前重量小於最小稱重重量, 通常重量值显示0
                    uint8_t txChar[] = "NoLoad,W= 0\n";
                    Uart0_TxByte_Polling(txChar, sizeof(txChar));
                }
                else if (scales.status == BODY_SCALES_STATE_LOADUP)
                {
                    // 有上稱動作/鎖定狀態下加載解鎖重量,重量由0點變成>最小稱重重量，通常重量值显示为实际重量
                    uint8_t txChar[] = "LoadUp,W=      \n";
                    Int32ToString(scales.weightReally, &txChar[10]);
                    Uart0_TxByte_Polling(txChar, sizeof(txChar));
                    haltTimeMs = TIMEOUT_MAX - TIMEOUT_10S;    // update halt time
                }
                else if (scales.status == BODY_SCALES_STATE_LOAD_OK)
                {
                    // 完成一次稱重測量，通常重量值显示为实际重量
                    uint8_t txChar[] = "LoadOk,W=      \n";
                    Int32ToString(scales.weightStable, &txChar[10]);
                    Uart0_TxByte_Polling(txChar, sizeof(txChar));
                }
                else if (scales.status == BODY_SCALES_STATE_LOAD_FIX)
                {
                    // 完成一次稱重測量后重量穩定沒有解鎖，通常重量值显示为实际重量
                    uint8_t txChar[] = "LoadFix,W=      \n";
                    Int32ToString(scales.weightStable, &txChar[11]);
                    Uart0_TxByte_Polling(txChar, sizeof(txChar));
                }
                else if (scales.status == BODY_SCALES_STATE_LOAD_DOWN)
                {
                    // 下秤動作，通常重量值显示为实际重量
                    uint8_t txChar[] = "LoadDown,W=      \n";
                    Int32ToString(scales.weightStable, &txChar[12]);
                    Uart0_TxByte_Polling(txChar, sizeof(txChar));
                    haltTimeMs = TIMEOUT_MAX - TIMEOUT_8S;    // update halt time
                }
                else if (scales.status == BODY_SCALES_STATE_OVERLOAD)
                {
                    // 超重,當前重量大於最大稱重重量，通常重量值显示为实际重量或错误提示码
                    uint8_t txChar[] = "OverLoad,W=      \n";
                    Int32ToString(scales.weightReally, &txChar[12]);
                    Uart0_TxByte_Polling(txChar, sizeof(txChar));
                    haltTimeMs = TIMEOUT_MAX - TIMEOUT_10S;    // update halt time
                }
                else if (scales.status == BODY_SCALES_STATE_AUTOON_SLOW)
                {
                    if (scalesStatusLast != scales.status)
                    {
                        WeightHardwareSetting(ADC24_RATE_10HZ);
                    }
                }
                else if (scales.status == BODY_SCALES_STATE_AUTOON_FAST)
                {

                }
                else if (scales.status == BODY_SCALES_STATE_AUTOON_PASS)    // 慢速ADC自動上稱判斷
                {
                    SystemEnterWakeUp();    // 系統喚醒
                    BodyScales_EnterWeighting(&scales, false);
                    uint8_t txChar[] = "WakeUp\n";
                    Uart0_TxByte_Polling(txChar, sizeof(txChar));
                }
                else if (scales.status == BODY_SCALES_STATE_AUTOON_FAIL)    // 自動上秤失敗
                {
                    SystemEnterHalt();    // 進入休眠
                }
            }
            scalesStatusLast = scales.status;
        }
    }
}

void WeightHardwareSetting(uint8_t sampling)
{
    // config AfePower
    Voreg_Cfg(VOREG_2V4);
    Voreg_Enable();
    // config Adc
    Adc24_Cfg_t Cfg;
    Cfg.sampleRate = sampling;
    Cfg.refVoltage = ADC24_REF_VOREG_AVSS;
    Cfg.gain       = ADC24_REFx0_5_GAINx128;
    Cfg.dcSet      = ADC24_DCSET_0V;
    Cfg.channel    = ADC24_CHP_AN0_N_AN1;
    Adc24_Cfg(&Cfg);
    Adc24_Enable();    // user adc interrupt
}

void SystemEnterHalt()
{
    haltTimeMs = TIMEOUT_MAX - TIMEOUT_30S;
    Uart0_Disable();
    Voreg_Disable();
    Adc24_Disable();
    Wdt_Enable(WDT_TIME_1000MS);
    haltTimeMs = 0;
    GCC_HALT();
}

void SystemEnterWakeUp()
{
    haltTimeMs = TIMEOUT_MAX - TIMEOUT_30S;
    Uart0_Enable();
    Voreg_Enable();
    Adc24_Enable();
    Wdt_Enable(WDT_TIME_125MS);
}

//@----------Timebase0 中斷入口函數-----------@
DEFINE_ISR(Timebase0_ISR, 0x20)
{
    haltTimeMs++;
}