/************************************************************
* Infrared Temperature Sensing Module BMH06203 example code
* HT-IDE3000 version: 8.05
* MCU DEV Board: ESK-303
* Wire Connection:
*   BMH06203 module pin 1 (VDD) connects to ESK-303 pin 5V
*   BMH06203 module pin 2 (SCL) connects to ESK-303 pin PA7
*   BMH06203 module pin 3 (SDA) connects to ESK-303 pin PA6
*   BMH06203 module pin 4 (GND) connects to ESK-303 pin GND

* Created on 2021/9/29 by Best Modules Corp.
*************************************************************/

#include "HT66F2390.h"
#include "MasterIIC.c"

// RAM command
#define AMB_CMD 0x08              // Ambient temperature command
#define OBJ_CMD 0x09              // Object temperature command
#define BODY_CMD 0x0A             // Body temperature command

// EEPROM command
#define MODE_CMD 0x28             // Mode choose
#define EMISSIVITY_CMD 0x29       // Emissivity setting
#define PWM_MIN_CMD 0x2A          // Set the minimum temperature in PWM mode(default 0, unit 0.1XC)
#define PWM_MAX_CMD 0x2B          // Set the maximum temperature in PWM mode(default 1000, unit 0.1XC)
#define IO_CMD 0x2C               // Set the temperature threshold in IO mode(default 0, unit 0.1XC)
#define SLEEP_CMD 0xFF            // Sleep mode setting
#define SLEEP_DATA 0x1234         // Sleep mode data

u16 I2CRead(u8);
void I2CWrite(u8,u16);
void Delay1ms(u16);

u16 Ambient,Object,Body,Emissivity;

void main()
{
	_wdtc=0xAB;
	
/*************************************************
Description: Set the emissivity 

Note: Power on again after changing the emissivity
*************************************************/ 

//	 I2CWrite(EMISSIVITY_CMD,0x5F);

/*************************************************
Description: Set to pwm mode(maximum=1000,minimum=100)

Note: Power on again after changing the PWM mode
*************************************************/

//	 I2CWrite(MODE_CMD,MODE_PWM_DATA);
// 	 I2CWrite(PWM_MIN_CMD,0x0064);
//	 I2CWrite(PWM_MAX_CMD,0x03E8);
  
/*************************************************
Description: Set to I/O mode(threshold=34.0XC)

Note: Power on again after changing the I/O mode
*************************************************/  

// 	I2CWrite(MODE_CMD,MODE_IO_CTR1_DATA);
// 	I2CWrite(IO_CMD,0x0154);

/*************************************************
Description: Set to I2C mode

Note: Power on again after changing the I2C mode
*************************************************/  

//  IIC_SCL=0;
//  Delay1ms(50);

	while(1) 
	{
		Emissivity=I2CRead(EMISSIVITY_CMD);		// Read emissivity value
		Delay1ms(100);
		
		Ambient=I2CRead(AMB_CMD);	// Read ambient temperature value
		Delay1ms(100);
	
		Object=I2CRead(OBJ_CMD);	// Read object temperature value
		Delay1ms(100);
	
		Body=I2CRead(BODY_CMD);		// Read body temperature value
		Delay1ms(100);
			
	}
}

/*************************************************
Description: Read command from RAM/EEPROM

Parameter:          
  command: the command to be executed

Return:             
  command is AMB_CMD : return ambient temperature value(unit 0.1XC)
  command is OBJ_CMD : return object temperature value(unit 0.1XC)
  command is BODY_CMD : return body temperature value(unit 0.1XC)
  command is MODE_CMD : return modeconfig value
  command is EMISSIVITY_CMD : return emissivity value
  command is PWM_MIN_CMD : return minimum temperature value(default 0, unit 0.1XC)
  command is PWM_MAX_CMD : return mmaximum temperature value(default 1000, unit 0.1XC)
  command is IO_CMD : return I/O threshold value(default 0, unit 0.1XC)         
*************************************************/
u16 I2CRead(u8 command)
{    
	return fun_IRSensor_ReadTemp(command);
}

/*************************************************
Description: Write command to EEPROM

Parameter:          
  command: The command to be executed
  value: The value to be written

Note: 

  Set modeconfig : command => MODE_CMD (default is I2C)
                   value   => MODE_I2C_DATA (Set the module to I2C mode)     
                   value   => MODE_PWM_DATA (Set the module to PWM mode)     
                   value   => MODE_IO_CTR0_DATA (Set the module to I/O mode (CTR=0))   
                              CTR = 0 : IO output low level when >= temperature threshold, IO output high level when < temperature threshold  
                   value   => MODE_IO_CTR1_DATA (Set the module to I/O mode (CTR=1))   
                              CTR = 1 : IO output High level when >= temperature threshold, IO output low level when < temperature threshold  
                      
  Set emissivity : command => EMISSIVITY_CMD
                   value   => Emissivity value
                   * If the emissivity is >= 100, the emissivity is 100%
                   * If the emissivity is 95, the emissivity is 95%
                   * If the emissivity is 50, the emissivity is 50%
                   
  Set sleep mode : command => SLEEP_CMD
                   value   => SLEEP_DATA
                   
--------------------------If modeconfig choose PWM--------------------------

  Set PWM minimum : command => PWM_MIN_CMD
                    value   => Minimum temperature value
                   
  Set PWM maximum : command => PWM_MAX_CMD
                    value   => Mmaximum temperature value
                    * If set PWM minimum to 100, PWM maximum to 1000,the temperature is 35XC at this time, 
                    * the PWM output is (350-100) / (1000-100) = 28% duty cycle Square wave.
                   
--------------------------If modeconfig choose IO--------------------------- 
                  
  Set I/O mode : command => IO_CMD
                   value => I/O threshold value
                   * If the temperature threshold IO is set to 450,the level will change when the temperature is ?=45.0XC                 
*************************************************/
void I2CWrite(u8 command ,u16 value)
{
	fun_IRSensor_SendCOM(command,value);
}

void Delay1ms(u16 delay)							//Delay1ms
{	
	u16 i;											
	for(i=0;i<delay;i++) GCC_DELAY(2000);			//GCC_DELAY(0.5us)
}	