/*****************************************************************
  File:             BM22S3221-1.cpp
  Author:           BEST MODULES CORP.
  Description:      Communication and operation function with module
  History：
  V1.0.2-- 2nd version；2022-10-20；Arduino IDE : v1.8.13
******************************************************************/
#include  "BM22S3221-1.h"
uint8_t receiveBuffer[32]={0};

/**********************************************************
Description: Select the hardware serial port you need to use
Parameters:  *theSerial：hardware serial 
             BMduino optional:serial(default) serial1/seria2/seria3/seria4
             UNO optional:serial(default)  
**********************************************************/
BM22S3221_1::BM22S3221_1(uint8_t statusPin,HardwareSerial*theSerial)
{
  _serial = theSerial;
  _softSerial = NULL;
  _statusPin = statusPin;
}
/**********************************************************
Description: Select the software serial port RX TX you need to use
Parameters:  rxPin:RX pin on the development board
             txPin:TX pin on the development board
**********************************************************/
BM22S3221_1::BM22S3221_1(uint8_t statusPin,uint8_t rxPin, uint8_t txPin)
{
  _serial = NULL;
  _statusPin = statusPin;
  _rxPin = rxPin;
  _txPin = txPin;
  _softSerial = new SoftwareSerial(_rxPin, _txPin);
}
/**********************************************************
Description: Set serial baud rate
Parameters:  uartBaud：9600(default)
Return:      none
**********************************************************/
void BM22S3221_1::begin()
{
  if (_softSerial != NULL) {_softSerial->begin(UART_BAUD);}
  else  {_serial->begin(UART_BAUD);}
}

/**********************************************************
U00
Description: reset the MCU in the module
Parameters:  none
Return:      0: success
             1: failed   
**********************************************************/
uint8_t BM22S3221_1::resetModule()
{ 
  uint8_t sum=0;
  uint8_t sendCmd[4] = {0xAF, 0x00, 0x00, 0x51};
  writeData(sendCmd, 4);
  delay(20);// wait module response
  readData(receiveBuffer);
  delay(80);// wait module to be stable.

  if (debug){
    Serial.println("U00: reset module");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}  
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return 0;
  else return  1;  
}
/**********************************************************
U01
Description: Query the FW version and production date.
             The FW version number and production date are both 8421 BCD code.
Parameters:  buff(12 bytes): Store the FW version and production date
Return:      0: success
             1: failed
**********************************************************/
uint8_t BM22S3221_1::getVerDate(uint8_t buff[])
{
  uint8_t sum=0;  
  uint8_t sendCmd[4] = {0xAD, 0x00, 0x00, 0x53};
  writeData(sendCmd, 4);
  delay(30); //wait module response
  readData(buff);
  if (debug){
    Serial.println("U01: get version and date");
    for (uint8_t i=0; i<12; i++)  {Serial.println(buff[i], HEX);}
  }  
  for (uint8_t i=0; i<11; i++) {sum += buff[i];}  
  if (buff[11] == (uint8_t)(~sum+1)) return 0;
  else  return   1;  
}
/**********************************************************
U03
Description: Get all current data of the module
Parameters:  buff(25byte): for storing the data
Return: 0: success
        1: failed
**********************************************************/
uint8_t BM22S3221_1::getInfoPackage(uint8_t buff[])
{
  uint8_t sum=0;
  uint8_t sendCmd[4] = {0xAC, 0x00, 0x00, 0x54};
  writeData(sendCmd, 4);
  delay(80);//wait module response
  readData(buff);
  if (debug){
    Serial.println("U03: get module information");
    for (uint8_t i=0; i<34; i++)  {
      Serial.print(i); 
      Serial.print(" : 0x");  
      Serial.println(buff[i], HEX);
    }
  }  
  for (uint8_t i=0; i<31; i++) {sum += buff[i];}  
  if (buff[31] == (uint8_t)(~sum+1)) return 0;
  else return  1;
}
/**********************************************************
U04
Description: restore the module to factory default settings
Parameters:  none
Return:      0: success
             1: failed
NOTE: after executing this command, the module preheats again 
**********************************************************/
uint8_t BM22S3221_1::restoreDefault()
{
  uint8_t sum=0;
  uint8_t sendCmd[4] = {0xA0, 0x00, 0x00, 0x60};
  writeData(sendCmd, 4);
  delay(100);//wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("U04: restore factory default settings");
    for (uint8_t i=0; i<8; i++)  {Serial.println(receiveBuffer[i], HEX);}
  }
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}  
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return 0;
  else return  1;
}
/**********************************************************
R01
Description: Query whether the UART auto-Tx is enabled
Parameters:  none
Return:      0: auto-Tx disabled
             8: auto-Tx enabled
             1: failed
NOTE: autoTx should be disabled before use this command 
**********************************************************/
uint8_t BM22S3221_1::isAutoTx()
{
  uint8_t sum = 0;
  uint8_t sendCmd[4] = {0xD0, 0x1B, 0, 0x15}; 
  writeData(sendCmd, 4);
  delay(30);   // wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("R01: get auto-tx status");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return  receiveBuffer[6];
  else return  1;  
}
/**********************************************************
R02
Description: Get the alarm output level of the STATUS pin
Parameters: None
Return: 8 : STATUS pin is high when alarming, low when not alarming
        0 : STATUS pin is low when alarming, high when not alarming
        1 : failed
NOTE: autoTx should be disabled before use this command 
**********************************************************/
uint8_t BM22S3221_1::getStatusPinActiveMode()
{
  uint8_t sum = 0;	
  uint8_t sendCmd[4] = {0xD0, 0x1C, 0x00, 0x14};  
  writeData(sendCmd, 4);
  delay(30); // wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("R02: get alarm level");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return  receiveBuffer[6];
  else return  1;  
}
/**********************************************************
R03
Description: Query the status of the module 
Parameters: None
Return:    the status of the module (refer to datasheet for meaning of each bit)
           0xFF: failed
NOTE: autoTx should be disabled before use this command 
**********************************************************/
uint8_t BM22S3221_1::getStatus()
{
  uint8_t sum = 0;		
  uint8_t sendCmd[4] = {0xD2, 0x88, 0x00, 0xA6};  
  writeData(sendCmd, 4);
  delay(30); // wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("R03: get status");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return  receiveBuffer[6];
  else return 0xFF;    
}
/**********************************************************
R04
Description: Read CO AD value high byte
Parameters: None
Return: AD value (high byte) for CO concentration
		0xFF: failed
NOTE: autoTx should be disabled before use this command 
**********************************************************/
uint8_t BM22S3221_1::getADValueH()
{
  uint8_t sum = 0;		
  uint8_t sendCmd[4] = {0xD2, 0x80, 0x00, 0xAE};  
  writeData(sendCmd, 4);
  delay(30); // wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("R04: read AD value high-byte");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return  receiveBuffer[6];
  else return 0xFF;  
}  
/**********************************************************
R05
Description: Read CO AD value low byte
Parameters: None
Return: AD value (low byte) for CO concentration
		0: failed
NOTE: autoTx should be disabled before use this command 
**********************************************************/
uint8_t BM22S3221_1::getADValueL()
{
  uint8_t sum = 0;		
  uint8_t sendCmd[4] = {0xD2, 0x81, 0x00, 0xAD};  
  writeData(sendCmd, 4);
  delay(30); // wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("R05: read AD value low-byte");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return  receiveBuffer[6];
  else return 0;  
}  
/**********************************************************
R06
Description: Read the power-on reference value high byte
Parameters: None
Return: power-on reference value (high byte) of CO concentration
		0xFF: failed
NOTE: autoTx should be disabled before use this command 
**********************************************************/
uint8_t BM22S3221_1::getRefValueH()
{
  uint8_t sum = 0;		
  uint8_t sendCmd[4] = {0xD2, 0x82, 0x00, 0xAC};  
  writeData(sendCmd, 4);
  delay(30); // wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("R06: read Ref value high-byte");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return  receiveBuffer[6];
  else return 0xFF;  
}  
/**********************************************************
R07
Description: Read the power-on reference value low byte
Parameters: None
Return: power-on reference value (low byte) of CO concentration
		0: failed
NOTE: autoTx should be disabled before use this command 
**********************************************************/
uint8_t BM22S3221_1::getRefValueL()
{
  uint8_t sum = 0;		
  uint8_t sendCmd[4] = {0xD2, 0x83, 0x00, 0xAB};  
  writeData(sendCmd, 4);
  delay(30); // wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("R07: read Ref value low-byte");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return  receiveBuffer[6];
  else return 0;  
}  
/**********************************************************
R08
Description: Read CO concentration value high byte
Parameters: None
Return: CO concentration value (high byte) in ppm
		0xFF: failes
NOTE: autoTx should be disabled before use this command 
**********************************************************/
uint8_t BM22S3221_1::getCOValueH()
{
  uint8_t sum = 0;		
  uint8_t sendCmd[4] = {0xD2, 0x84, 0x00, 0xAA};  
  writeData(sendCmd, 4);
  delay(30); // wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("R08: read CO value high-byte");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return  receiveBuffer[6];
  else return 0xFF;  
} 
/**********************************************************
R09
Description: Read CO concentration value low byte
Parameters: None
Return: CO concentration value (low byte) in ppm
		0xFF: failes
NOTE: autoTx should be disabled before use this command 
**********************************************************/ 
uint8_t BM22S3221_1::getCOValueL()
{
  uint8_t sum = 0;		
  uint8_t sendCmd[4] = {0xD2, 0x85, 0x00, 0xA9};  
  writeData(sendCmd, 4);
  delay(30); // wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("R09: read CO value low-byte");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return  receiveBuffer[6];
  else return 0xFF;  
}  
/**********************************************************
R10
Description: Get the alarm threshold high byte
Parameters: None
Return: the alarm threshold value (high byte), in ppm. Default 180 ppm (=0xB4)
		0xFF: failed
NOTE: autoTx should be disabled before use this command 
**********************************************************/
uint8_t BM22S3221_1::getAlarmThresholdH()
{
  uint8_t sum = 0;		
  uint8_t sendCmd[4] = {0xD0, 0x0C, 0x00, 0x24};  
  writeData(sendCmd, 4);
  delay(30); // wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("R10: read alarm threshold high-byte");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return  receiveBuffer[6];
  else return 0xFF;  
} 
/**********************************************************
R11
Description: Get the alarm threshold low byte
Parameters: None
Return: the alarm threshold value (low byte), in ppm. Default 180 ppm (=0xB4)
		0xFF: failed
NOTE: autoTx should be disabled before use this command 
**********************************************************/
uint8_t BM22S3221_1::getAlarmThresholdL()
{
  uint8_t sum = 0;		
  uint8_t sendCmd[4] = {0xD0, 0x0D, 0x00, 0x23};  
  writeData(sendCmd, 4);
  delay(30); // wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("R11: read alarm threshold low-byte");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return  receiveBuffer[6];
  else return 0xFF;  
} 
/**********************************************************
R12
Description: Get the exit alarm threshold.When CO concentration is lower than the threshold value, the alarm stops. Default 55 ppm (=0x37)
Parameters: None
Return: the CO concentration value to stop alarming, in ppm.
		0xFF: failed
NOTE: autoTx should be disabled before use this command 
**********************************************************/
uint8_t BM22S3221_1::getExitAlarm()
{
  uint8_t sum = 0;		
  uint8_t sendCmd[4] = {0xD0, 0x0E, 0x00, 0x22};  
  writeData(sendCmd, 4);
  delay(30); // wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("R12: read Exit Alarm Threshold");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return  receiveBuffer[6];
  else return 0xFF;  
} 

/**********************************************************
W01
Description: Enable/Disable sending out device information automatically via UART
Parameters: state=08, enable (default)
            state=0, disable 
Return:      0: success
             1: failed
**********************************************************/
uint8_t BM22S3221_1::setAutoTx(uint8_t state)
{
  uint8_t sum=0;
  uint8_t sendCmd[4] = {0xE0, 0x1B, 0, 0};  
  sendCmd[2] = state;
  sum = sendCmd[0] + sendCmd[1] + sendCmd[2];
  sendCmd[3] = ~sum + 1;
  writeData(sendCmd, 4);
  delay(50);   // wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("W01: Enable/Disable AutoTx");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  sum = 0;
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return 0;
  else return  1;      
}
/**********************************************************
W02
Description: set alarm output (status pin) active level
Parameters: state=08, status pin is high when alarming, low when not alarming (default)
            state=0, status pin is low when alarming, high when not alarming
Return:      0: success
             1: failed
**********************************************************/
uint8_t BM22S3221_1::setStatusPinActiveMode(uint8_t state)
{
  uint8_t sum=0;
  uint8_t sendCmd[4] = {0xE0, 0x1C, 0, 0};  
  sendCmd[2] = state;
  sum = sendCmd[0] + sendCmd[1] + sendCmd[2];
  sendCmd[3] = ~sum + 1;
  writeData(sendCmd, 4);
  delay(50);   // wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("W02: set staus pin level");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  sum = 0;
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return 0;
  else return  1;        
}
/**********************************************************
W03
Description: Set the alarm threshold value high byte(in ppm). Default 4500 ppm = 0x1194
            When CO concentration is higher than the threshold value, alarm is set.
Parameters: alarm threshold value high byte 
Return: 0: success
        1: failed
**********************************************************/
uint8_t BM22S3221_1::setAlarmThresholdH(uint8_t value)
{
  uint8_t sum=0;
  uint8_t sendCmd[4] = {0xE0, 0x0C, 0, 0};  
  sendCmd[2] = value;
  sum = sendCmd[0] + sendCmd[1] + sendCmd[2];
  sendCmd[3] = ~sum + 1;
  writeData(sendCmd, 4);
  delay(50);   // wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("W03: set alarm threshold high-byte");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  sum = 0;
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return 0;
  else return  1;        
}
/**********************************************************
W04
Description: Set the alarm threshold value low byte(in ppm). Default 4500 ppm = 0x1194
            When CO concentration is higher than the threshold value, alarm is set.
Parameters: alarm threshold value low byte 
Return: 0: success
        1: failed
**********************************************************/
uint8_t BM22S3221_1::setAlarmThresholdL(uint8_t value)
{
  uint8_t sum=0;
  uint8_t sendCmd[4] = {0xE0, 0x0D, 0, 0};  
  sendCmd[2] = value;
  sum = sendCmd[0] + sendCmd[1] + sendCmd[2];
  sendCmd[3] = ~sum + 1;
  writeData(sendCmd, 4);
  delay(50);   //wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("W04: set alarm threshold low-byte");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  sum = 0;
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return 0;
  else return  1;        
}
/**********************************************************
W05
Description: Set the exit alarm threshold value (in ppm). Default 55 ppm = 0x37
            When CO concentration is lower than the threshold value, alarm stops.
Parameters:  the exit-alarm threshold value
Return:      0: success
             1: failed
**********************************************************/
uint8_t BM22S3221_1::setExitAlarmThreshold(uint8_t value)
{
  uint8_t sum=0;
  uint8_t sendCmd[4] = {0xE0, 0x0E, 0, 0};  
  sendCmd[2] = value;
  sum = sendCmd[0] + sendCmd[1] + sendCmd[2];
  sendCmd[3] = ~sum + 1;
  writeData(sendCmd, 4);
  delay(50);   // wait module response
  readData(receiveBuffer);
  if (debug){
    Serial.println("W05: set exit-alarm threshold");
    for (uint8_t i=0; i<8; i++){Serial.println(receiveBuffer[i], HEX);}
  }
  sum = 0;
  for (uint8_t i=0; i<7; i++) {sum += receiveBuffer[i];}
  if (receiveBuffer[7] == (uint8_t)(~sum+1)) return 0;
  else return  1;        
}
/**********************************************************
Description: Read the data automatically output by the module
Parameters:  buff: buffer for storing the received data 
Return:      0: success
             1: failed
**********************************************************/
uint8_t BM22S3221_1::autoRx(uint8_t buff[])
{
  uint8_t sum=0,num=0;
  clear_UART_FIFO();
  while (num < 32){
    if (_softSerial != NULL) {num = _softSerial->available();}
    else if (_serial != NULL){num = _serial->available();}
  }
  for (uint8_t i=0;i<32;i++){
    if (_softSerial != NULL) {buff[i] = _softSerial->read();}
    else if (_serial != NULL){buff[i] = _serial->read();}
  }
  for(uint8_t i=0;i<31;i++){sum += buff[i];}
  sum = ~sum + 1;
  if (debug){
    Serial.print("Auto Rx  ");
    Serial.println(num);
    for(uint8_t i=0;i<31;i++){
      Serial.print(i);
      Serial.print(": 0x");
      Serial.println(buff[i], HEX);
    }
    Serial.println(sum, HEX);
    Serial.println("");  
  }
  if ((sum == buff[31]) && (buff[0]==0xAA)){return 0;}
  else return 1;
}
/**********************************************************
Description: UART writeData
             Automatically determine whether a hardware or software serial  is used
Parameters:  buff: array where the data is stored to send
             length: the number of bytes to be sent from the array.
Return:      none
**********************************************************/
void  BM22S3221_1::writeData(uint8_t buff[], uint8_t length)
{
  clear_UART_FIFO();  
  if (_softSerial != NULL) {_softSerial->write(buff, length);}  
  else {_serial->write(buff, length);}
}
/**********************************************************
Description: Clear UART FIFO
Parameters: None
Return: None
**********************************************************/
void BM22S3221_1::clear_UART_FIFO()
{
  if (_softSerial != NULL)
  {
    while (_softSerial->available() > 0) {_softSerial->read();}
  }
  if (_serial != NULL)
  {
    while (_serial->available() > 0) {_serial->read();}
  }
}
/**********************************************************
Description: UART readData
Parameters:       buff:Variables for storing Data to be read
             datalength:Length of data plus command
Return:      none
Others:
**********************************************************/
void  BM22S3221_1::readData(uint8_t buff[])
{
  uint8_t i=0;
  if (_softSerial != NULL)
  {
    while(_softSerial->available()>0)
    {
      buff[i] = _softSerial->read();
      i++;
    }
  }
  else
  {
    while(_serial->available()>0)
    {
      buff[i] = _serial->read();
      i++;
    }
  }
}
