/*********************************************************************************************************//**
 * @file    BMD26M088_HT32/colorful/main.c
 * @version V1.0.1
 * @date    2024-08-01
 * @brief   Main program.
 *************************************************************************************************************
 * @attention
 *
 * Firmware Disclaimer Information
 *
 * 1. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, which is supplied by Holtek Semiconductor Inc., (hereinafter referred to as "HOLTEK") is the
 *    proprietary and confidential intellectual property of HOLTEK, and is protected by copyright law and
 *    other intellectual property laws.
 *
 * 2. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, is confidential information belonging to HOLTEK, and must not be disclosed to any third parties
 *    other than HOLTEK and the customer.
 *
 * 3. The program technical documentation, including the code, is provided "as is" and for customer reference
 *    only. After delivery by HOLTEK, the customer shall use the program technical documentation, including
 *    the code, at their own risk. HOLTEK disclaims any expressed, implied or statutory warranties, including
 *    the warranties of merchantability, satisfactory quality and fitness for a particular purpose.
 *
 * <h2><center>Copyright (C) Holtek Semiconductor Inc. All rights reserved</center></h2>
 ************************************************************************************************************/

/* Includes ------------------------------------------------------------------------------------------------*/
#include "ht32.h"
#include "ht32_board.h"
#include "ht32_board_config.h"

#include "BMD26M088.h"

/* Private types -------------------------------------------------------------------------------------------*/
typedef struct RGB_DATA
{
  uint8_t r;
  uint8_t g;
  uint8_t b;
} rgb_def;

/* Private function prototypes -----------------------------------------------------------------------------*/
rgb_def get_color(int16_t param);
void colorful__MainRoutine(void);

/* Global variables ----------------------------------------------------------------------------------------*/
int16_t flow = 0;

/* Global functions ----------------------------------------------------------------------------------------*/
/*********************************************************************************************************//**
  * @brief  Main program.
  * @retval None
  ***********************************************************************************************************/
int main(void)
{
  /*
    !!! NOTICE !!!
    BMD26M088 Module I2C slave address should be one of the following values (7-bit Addressing Mode):
    (Please check the jumper on the back of the module, and set the corresponding I2C slave address)
    - BMD26M088_I2C_ADDRESS_VDD       : 0x67 (Factory Default)
    - BMD26M088_I2C_ADDRESS_GND       : 0x64
    - BMD26M088_I2C_ADDRESS_SCL       : 0x65
    - BMD26M088_I2C_ADDRESS_SDA       : 0x66
    - BMD26M088_I2C_ADDRESS_BOARDCAST : 0x2E (Boardcast Address)
  */
  BMD26M088_Init(BMD26M088_I2C_ADDRESS_VDD, WIRE_CLOCK_SPEED);

  BMD26M088_setBrightness(BMD26M088_I2C_ADDRESS_VDD, 0xFF); /* Set global brightness, Max                   */

  BMD26M088_setCurrent(BMD26M088_I2C_ADDRESS_VDD, BMD26M088_CCR_6MA); /* Set drive current 6mA              */

  while (1)                           /* Infinite loop                                                      */
  {
    colorful__MainRoutine();
  }
}

/*********************************************************************************************************//**
  * @brief  colorful Main Routine.
  * @retval None
  ***********************************************************************************************************/
void colorful__MainRoutine(void)
{
  uint8_t i;
  rgb_def rgb;

  for (i = 0; i < 8; i++)
  {
    rgb = get_color(flow + i * (420 / 16)); /* Get color to update RGB                                      */

    BMD26M088_writeRow(BMD26M088_I2C_ADDRESS_VDD, i, rgb.r, rgb.g, rgb.b); /* Update a row RGB              */
  }

  flow++;
  if (flow > 420)
  {
    flow = 0;
  } /* flowing                                                                                              */
}

/*********************************************************************************************************//**
  * @brief  Change color through calculation.
  * @retval None
  ***********************************************************************************************************/
rgb_def get_color(int16_t param)
{
  rgb_def rgb;

  if (param < 0)
  {
    param += 420;
  }
  if (param > 419)
  {
    param -= 420;
  }

  if (param < 60) /* Step 1: Red -> Yellow                                                                  */
  {
    rgb.r = 240;
    rgb.g = 4 * param;
    rgb.b = 0;
  }
  else if (param < 120) /* Step 2: Yellow -> Green                                                          */
  {
    rgb.r = 240 - (4 * (param - 60));
    rgb.g = 240;
    rgb.b = 0;
  }
  else if (param < 180) /* Step 3: Green -> Cyan                                                            */
  {
    rgb.r = 0;
    rgb.g = 240;
    rgb.b = 4 * (param - 120);
  }
  else if (param < 240) /* Step 4: Cyan close to Blue                                                       */
  {
    rgb.r = 0;
    rgb.g = 240 - (3 * (param - 180));
    rgb.b = 240;
  }
  else if (param < 300) /* Step 5: Blue -> Purple                                                           */
  {
    rgb.r = (param - 240);
    rgb.g = 60 - (param - 240);
    rgb.b = 240 - (2 * (param - 240));
  }
  else if (param < 360) /* Step6: Purple -> Pink                                                            */
  {
    rgb.r = 60 + (param - 300);
    rgb.g = 0;
    rgb.b = 120;
  }
  else if (param < 420) /* Step7: Pink -> Red                                                               */
  {
    rgb.r = 120 + (2 * (param - 360));
    rgb.g = 0;
    rgb.b = 120 - (2 * (param - 360));
  }

  return rgb;
}

#if (HT32_LIB_DEBUG == 1)
/*********************************************************************************************************//**
  * @brief  Report both the error name of the source file and the source line number.
  * @param  filename: pointer to the source file name.
  * @param  uline: error line source number.
  * @retval None
  ***********************************************************************************************************/
void assert_error(u8* filename, u32 uline)
{
  /*
     This function is called by IP library that the invalid parameters has been passed to the library API.
     Debug message can be added here.
     Example: printf("Parameter Error: file %s on line %d\r\n", filename, uline);
  */

  while (1)
  {
  }
}
#endif

/* Private functions ---------------------------------------------------------------------------------------*/
