/*********************************************************************************************************//**
 * @file    BMV23M001_HT32/src/BMV23M001.c
 * @version V1.0.1
 * @date    2024-08-27
 * @brief   The function of BMV23M001 driver.
 *************************************************************************************************************
 * @attention
 *
 * Firmware Disclaimer Information
 *
 * 1. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, which is supplied by Holtek Semiconductor Inc., (hereinafter referred to as "HOLTEK") is the
 *    proprietary and confidential intellectual property of HOLTEK, and is protected by copyright law and
 *    other intellectual property laws.
 *
 * 2. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, is confidential information belonging to HOLTEK, and must not be disclosed to any third parties
 *    other than HOLTEK and the customer.
 *
 * 3. The program technical documentation, including the code, is provided "as is" and for customer reference
 *    only. After delivery by HOLTEK, the customer shall use the program technical documentation, including
 *    the code, at their own risk. HOLTEK disclaims any expressed, implied or statutory warranties, including
 *    the warranties of merchantability, satisfactory quality and fitness for a particular purpose.
 *
 * <h2><center>Copyright (C) Holtek Semiconductor Inc. All rights reserved</center></h2>
 ************************************************************************************************************/



/* Includes ------------------------------------------------------------------------------------------------*/
#include "BMV23M001.h"

/* Settings ------------------------------------------------------------------------------------------------*/
/* Private types -------------------------------------------------------------------------------------------*/
/* Private constants ---------------------------------------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------------------------------------*/
/* Global variables ----------------------------------------------------------------------------------------*/
uint32_t gBMV23M001_WIRE = BMV23M001_WIRE;
/* Private variables ---------------------------------------------------------------------------------------*/

/* Global functions ----------------------------------------------------------------------------------------*/
/*********************************************************************************************************//**
 * @brief 	module wire number select.
 * @param 	wire_number: select wire number.
 * @retval 	None
 ************************************************************************************************************/
BMV23M001_selStatus BMV23M001_selWire(uint32_t wire_number)
{
  if(CHECK_WIREn(wire_number) == -1)
  {
    return BMV23M001_FAILURE;
  }

  gBMV23M001_WIRE = wire_number;
  return BMV23M001_SUCCESS;
}
/*********************************************************************************************************//**
 * @brief  Module initialization using I2C communication.
 * @param  void
 * @retval void
 ************************************************************************************************************/
void BMV23M001_Init(void)
{
	CKCU_PeripClockConfig_TypeDef CKCUClock = {{ 0 }};
	CKCUClock.Bit.AFIO = 1;
	CKCUClock.Bit.PB = 1;
	CKCUClock.Bit.ADC0 = 1;
	CKCU_PeripClockConfig(CKCUClock, ENABLE);
	
	AFIO_GPxConfig(BMV23M001_STAPIN_GPIO_ID, BMV23M001_STAPIN_AFIO_PIN, AFIO_FUN_GPIO);
	GPIO_PullResistorConfig(BMV23M001_STAPIN_GPIO_PORT, BMV23M001_STAPIN_GPIO_PIN, GPIO_PR_UP);
	GPIO_DirectionConfig(BMV23M001_STAPIN_GPIO_PORT, BMV23M001_STAPIN_GPIO_PIN, GPIO_DIR_IN);
	GPIO_InputConfig(BMV23M001_STAPIN_GPIO_PORT, BMV23M001_STAPIN_GPIO_PIN, ENABLE);
	
	AFIO_GPxConfig(BMV23M001_GATEPIN_GPIO_ID, BMV23M001_GATEPIN_AFIO_PIN, AFIO_FUN_GPIO);
	GPIO_PullResistorConfig(BMV23M001_GATEPIN_GPIO_PORT, BMV23M001_GATEPIN_GPIO_PIN, GPIO_PR_UP);
	GPIO_DirectionConfig(BMV23M001_GATEPIN_GPIO_PORT, BMV23M001_GATEPIN_GPIO_PIN, GPIO_DIR_IN);
	GPIO_InputConfig(BMV23M001_GATEPIN_GPIO_PORT, BMV23M001_GATEPIN_GPIO_PIN, ENABLE);
	
	AFIO_GPxConfig(GPIO_PA,  AFIO_PIN_1, AFIO_FUN_ADC0);					 
	AFIO_GPxConfig(GPIO_PA,  AFIO_PIN_2, AFIO_FUN_ADC0);					 

	CKCU_SetADCnPrescaler(CKCU_ADCPRE_ADC0, CKCU_ADCPRE_DIV64);   
	
	ADC_RegularGroupConfig(HT_ADC0, ONE_SHOT_MODE, 2, 1);					 
	
	ADC_SamplingTimeConfig(HT_ADC0, 0);													 
	
	ADC_RegularChannelConfig(HT_ADC0, ADC_CH_1, 0);								 
	ADC_RegularChannelConfig(HT_ADC0, ADC_CH_2, 1);								 

	ADC_RegularTrigConfig(HT_ADC0, ADC_TRIG_SOFTWARE);						 

	ADC_Cmd(HT_ADC0, ENABLE);
	
	I2CMaster_Init(gBMV23M001_WIRE, BMV23M001_MASTERADDR, BMV23M001_CLKFREQ);
}
/*********************************************************************************************************//**
 * @brief  Obtain the ADC value of pin A0(PA1)
 * @param  void
 * @retval ADC value
 ************************************************************************************************************/
uint16_t BMV23M001_ADC_GetA0(void)
{
	ADC_SoftwareStartConvCmd(HT_ADC0, ENABLE); 
	while(ADC_GetFlagStatus(HT_ADC0,ADC_FLAG_SINGLE_EOC) != SET);	
	return ADC_GetConversionData(HT_ADC0,0);
}
/*********************************************************************************************************//**
 * @brief  Obtain the ADC value of pin A1(PA2)
 * @param  void
 * @retval ADC value
 ************************************************************************************************************/
uint16_t BMV23M001_ADC_GetA1(void)
{
	ADC_SoftwareStartConvCmd(HT_ADC0, ENABLE); 
	while(ADC_GetFlagStatus(HT_ADC0,ADC_FLAG_SINGLE_EOC) != SET);	
	return ADC_GetConversionData(HT_ADC0,1);
}
/*********************************************************************************************************//**
 * @brief  	get STA Pin(D22/PB12) Status
 * @param  	void
 * @retval 	STA pin Status  
						@arg 0:STA pin output low level  press
						@arg 1:STA pin output high level   unpress 	
 ************************************************************************************************************/
uint8_t BMV23M001_getStatus()
{
	return GPIO_ReadInBit(BMV23M001_STAPIN_GPIO_PORT, BMV23M001_STAPIN_GPIO_PIN);
}
/*********************************************************************************************************//**
 * @brief  get GATE Pin(D2/PB0) Status
 * @param  void
 * @retval GATE pin Status
           0:INT output low level  press
	   1:INT output high level   unpress 	
 ************************************************************************************************************/
uint8_t BMV23M001_getGateStatus()
{
	return GPIO_ReadInBit(BMV23M001_GATEPIN_GPIO_PORT, BMV23M001_GATEPIN_GPIO_PIN);
}
/*********************************************************************************************************//**
 * @brief  	Read the sound status with IIC
 * @param  	void
 * @retval 	Sound Status:0:Sound quiet   1:Sound play 	3:Fail  
 ************************************************************************************************************/
uint8_t BMV23M001_readSoundStatus(void)
{
	uint8_t rxData[7] = {0};
	uint8_t soundCMD[5]={0xA5,0x5A,0x01,0x01,0x01};

	BMV23M001_writeBytes(soundCMD,5);
	if(BMV23M001_readBytes(rxData,6) == BMV23M001_READ_OK)
	{
		return rxData[4];
	}
	
	return StatusFAIL;
}
/*********************************************************************************************************//**
 * @brief  Read the envelope ADC with IIC
 * @param  void
 * @retval ADC Value  
 ************************************************************************************************************/
uint16_t BMV23M001_readSoundEnvelopeADC(void)
{
	uint8_t rxData[7] = {0};
	uint8_t ADCCMD[5]={0xA5,0x5A,0x01,0x02,0x02};

	BMV23M001_writeBytes(ADCCMD,5);
	if(BMV23M001_readBytes(rxData,7) == BMV23M001_READ_OK)
	{
		return rxData[4] * 256 + rxData[5];
	}		
	return 0;
}
/*********************************************************************************************************//**
 * @brief  	Get the version information with IIC
 * @param  	void   
 * @retval 	Communication status 
						@arg 0:Fail 
						@arg other:FWVer 
 ************************************************************************************************************/
uint16_t BMV23M001_getFWVer(void)
{
	uint8_t rxData[7] = {0};
	uint8_t FWVCMD[5]={0xA5,0x5A,0x01,0x03,0x3};

	BMV23M001_writeBytes(FWVCMD,5);
	if(BMV23M001_readBytes(rxData,7) == BMV23M001_READ_OK)
	{
		return rxData[4] * 256 + rxData[5];						
	}
	
	return 0;
}
/*********************************************************************************************************//**
 * @brief  	write data through IIC
 * @param  	wbuf:Variables for storing Data to be sent
						wlen:Length of data sent
 * @retval 	void  
 ************************************************************************************************************/
void BMV23M001_writeBytes(uint8_t wbuf[], uint8_t wlen)
{
	I2CMaster_Status_Enum I2CResult = I2C_MASTER_GOING;
	I2CMaster_Typedef gI2CMaster = {0x00};
		
	gI2CMaster.uSlaveAddress = BMV23M001_MASTERADDR;
	gI2CMaster.Tx.puBuffer = (uint8_t *)wbuf;
  gI2CMaster.Tx.uLength = wlen;
  gI2CMaster.uTimeout_us = 30000;
  gI2CMaster.isDynamicOnOff_I2C = FALSE;

  I2CResult = I2C_MASTER_GOING;
  I2CMaster_Write(gBMV23M001_WIRE,&gI2CMaster);
  do {
      I2CResult = I2CMaster_GetStatus(gBMV23M001_WIRE);
  } while (I2CResult == I2C_MASTER_GOING);

	_BMV23M001_delay(1);
}
/*********************************************************************************************************//**
 * @brief  	read data through IIC
 * @param  	rbuf:Variables for storing Data to be obtained
						rlen:Length of data to be obtained
 * @retval 	BMV23M001_READ_OK/BMV23M001_READ_ERROR/BMV23M001_NO_ACK/BMV23M001_TIMEOUT_ERROR 
 ************************************************************************************************************/
uint8_t BMV23M001_readBytes(uint8_t rbuf[], uint8_t rlen)
{
	uint8_t i = 0;
	uint8_t checkSum = 0;
	
	I2CMaster_Status_Enum I2CResult = I2C_MASTER_GOING;
	I2CMaster_Typedef I2CMaster = { 0 };
	
	I2CMaster.uSlaveAddress = BMV23M001_MASTERADDR;
	I2CMaster.Rx.puBuffer = (uint8_t *)rbuf;
	I2CMaster.Rx.uLength = rlen;
	I2CMaster.uTimeout_us = 30000;
	I2CMaster.isDynamicOnOff_I2C = FALSE;

	I2CMaster_Read(gBMV23M001_WIRE, &I2CMaster);
	
	do {
		I2CResult = I2CMaster_GetStatus(gBMV23M001_WIRE);
	} while (I2CResult == I2C_MASTER_GOING);

	_BMV23M001_delay(1);
	for(i = 0;i < rlen - 1;i++)
	{
		checkSum = checkSum + rbuf[i];
	}
	if(rbuf[i] != checkSum)
	{
		return BMV23M001_READ_ERROR;
	}
	return BMV23M001_READ_OK;
}

/*********************************************************************************************************//**
 * @brief  delay ms.
 * @param  count
 * @retval void
 ************************************************************************************************************/
void _BMV23M001_delay(vu32 count)
{
  count = SystemCoreClock / 8000 * count;
  while(count--);
}
/* Private functions ---------------------------------------------------------------------------------------*/
