/*****************************************************************
File:             BM32S4021-1.cpp
Author:           BESTMODULES
Description:      UART communication with the BM32S4021_1 and obtain the corresponding value  
Version:          V1.0.1   -- 2024-01-08
******************************************************************/
#include "BM32S4021-1.h"
/**********************************************************
Description: Constructor
Parameters:  
             *theSerial: Wire object if your board has more than one UART interface      
                         parameter range:
                                         BMduino UNO: &Serial、&Serial1、&Serial2、&Serial3、&Serial4
                                         Arduino UNO：&Serial
Return:          
Others:     
**********************************************************/
BM32S4021_1::BM32S4021_1(uint8_t intPin,HardwareSerial *theSerial)
{
     _softSerial = NULL;
     _intPin = intPin;
     _hardSerial = theSerial;
}
/**********************************************************
Description: Constructor
Parameters:  
             rxPin: Receiver pin of the UART
             txPin: Send signal pin of UART         
Return:          
Others:   
**********************************************************/
BM32S4021_1::BM32S4021_1(uint8_t intPin,uint8_t rxPin,uint8_t txPin)
{
    _hardSerial = NULL;
     _intPin = intPin;
    _rxPin = rxPin;
    _txPin = txPin;
    _softSerial = new SoftwareSerial(_rxPin,_txPin);
}

/**********************************************************
Description: Module Initial
Parameters:    
Return:          
Others:   If the hardware UART is initialized, the _softSerial 
          pointer is null, otherwise it is non-null       
**********************************************************/
void BM32S4021_1::begin()
{
    delay(500);
    pinMode(_intPin,INPUT_PULLUP);
    if(_softSerial != NULL)
    {
     _softSerial->begin(9600); 
    }
    else
    {
      _hardSerial->begin(9600);
    }
}

/**********************************************************
Description: Get INT Status
Parameters:         
Return:     Returns the INT Status  0:INT output low level 
                                    1:INT output high level  
Others:     
**********************************************************/
uint8_t BM32S4021_1::getINT()
{
     return (digitalRead(_intPin));
}

/**********************************************************
Description: Get Gesture state
Parameters:   
Return:      irStatus:Gesture state
                 Bit6：Distance learning state
                       0: distance from learning success
                       1: distance learning failure   
                 Bit5：Whether there is a descent sign
                       0: flase
                       1: true 
                 Bit4：Whether there is a rise sign
                       0: flase
                       1: true 
                 Bit3：Whether there is a backslide gesture
                       0: flase
                       1: true  
                 Bit2：Whether there is a forward-sliding gesture
                       0: flase
                       1: true  
                 Bit1：Whether there is a right swipe gesture
                       0: flase
                       1: true 
                 Bit0：Whether there is a left swipe gesture
                       0: flase
                       1: true 
Others:      
**********************************************************/
uint8_t BM32S4021_1::getIRStatus()
{
    uint8_t sendBuf[5] = {0x55, 0x80, 0x02, 0x01, 0xD8};
    uint8_t buff[6] = {0};
    uint8_t  irStatus = 0;
    writeBytes(sendBuf,5);
    if(readBytes(buff,6)== CHECK_OK)
    {
     irStatus= buff[4];
    }
    delay(20);
    return irStatus;
}

/**********************************************************
Description: Example Query whether the information automatically output by the module is received
Parameters:         
Return:    
             true:  The information is automatically output by the received module
             false: The automatic output of the module was not received 
Others:      
**********************************************************/
bool  BM32S4021_1::isInfoAvailable()
{
  uint8_t checkSum = 0;
  if(_softSerial != NULL)
    {
      if(_softSerial->available() >= 6)
      {
       /* readBytes */
       for (uint8_t i = 0; i < 6; i++)
       {
         _recBuf[i] = _softSerial->read();
       }
       /* check Sum */
       for (uint8_t i = 0; i <5; i++)
       {
         checkSum += _recBuf[i];
       }
       if (checkSum ==  _recBuf[5])
       {
         if((_recBuf[0] == 0x55) && (_recBuf[1] == 0xC0) && (_recBuf[2] == 0x02))
         {
          return true;
         }
         else
         {
          return false; 
         }
       }
       else
       {
        return false; 
       }
      }
      else 
      {
        return   false;
      }
    }
    
   else
    {
      if(_hardSerial->available() >= 6)
      {
       /* readBytes */
       for (uint8_t i = 0; i < 6; i++)
       {
         _recBuf[i] = _hardSerial->read();
       }
       /* check Sum */
       for (uint8_t i = 0; i <5; i++)
       {
         checkSum += _recBuf[i];
       }
       if (checkSum ==  _recBuf[5])
       {
         if((_recBuf[0] == 0x55) && (_recBuf[1] == 0xC0) && (_recBuf[2] == 0x02))
         {
          return true;
         }
         else
         {
          return false; 
         }
       }
       else
       {
        return false;
       }
      }
      else 
      {
        return   false;
      }
    }
   
}

/**********************************************************
Description: Gets the gesture status of the module's active output
Parameters:         
Return:      _recBuf[4]：Gesture state  
                 Bit6：Distance learning state
                       0: distance from learning success
                       1: distance learning failure   
                 Bit5：Whether there is a descent sign
                       0: flase
                       1: true 
                 Bit4：Whether there is a rise sign
                       0: flase
                       1: true 
                 Bit3：Whether there is a backslide gesture
                       0: flase
                       1: true  
                 Bit2：Whether there is a forward-sliding gesture
                       0: flase
                       1: true  
                 Bit1：Whether there is a right swipe gesture
                       0: flase
                       1: true 
                 Bit0：Whether there is a left swipe gesture
                       0: flase
                       1: true 
Others:      
**********************************************************/
uint8_t  BM32S4021_1::readInfoPackage()
{
  return  _recBuf[4];
}

/**********************************************************
Description: Distance Learning
Parameters:         
Return:     Distance learning outcome
               0x00:Distance learning success 
               0x01:Distance learning failure  
Others:      
**********************************************************/
uint8_t BM32S4021_1::distanceLearning()
{
    uint8_t sendBuf[3] = {0x55, 0x20, 0x75};
    uint8_t buff[6] = {0};
    writeBytes(sendBuf,3);
    if(readBytes(buff,3)== CHECK_OK)
    {
     if(buff[1]== 0x7f)
     {
       if(readBytes(buff,6)== CHECK_OK)
      {
       delay(20);
       if(!(buff[4]&0x40)) 
        {
         return 0x00;
        }
      }
     }
    }
    delay(20);
    return 0x01;
}

/**********************************************************
Description: Cumulative continuous slip
Parameters:       
Return:      IRGesturenum: Cumulative frequency
Others:      
**********************************************************/
uint8_t BM32S4021_1::getIRGestureNum()
{
    uint8_t sendBuf[5] = {0x55, 0x80, 0x03, 0x01, 0xD9};
    uint8_t buff[6] = {0};
    uint8_t  IRGesturenum = 0;
    writeBytes(sendBuf,5);
    if(readBytes(buff,6)== CHECK_OK)
    {
     IRGesturenum= buff[4];
    }
    delay(20);
    return IRGesturenum;
}

/**********************************************************
Description: Get the version information 
Parameters:
Return:      FWVer: version number   
Others:      
**********************************************************/
uint16_t BM32S4021_1::getFWVer()
{ 
    uint8_t sendBuf1[5] = {0x55, 0x80, 0x00, 0x01,0xD6};
    uint8_t sendBuf2[5] = {0x55, 0x80, 0x01, 0x01,0xD7};
    uint8_t buff[6] = {0};
    uint8_t verh = 0,verl = 0;
    uint16_t FWVer = 0;
    writeBytes(sendBuf1,5);
    if(readBytes(buff,6)== CHECK_OK)
    {
    verl= buff[4];
    }
    delay(20);
    writeBytes(sendBuf2,5);
   if(readBytes(buff,6)== CHECK_OK)
   {
    verh= buff[4];
   }  
   FWVer = verl+ (verh<<8);
   delay(20);
   return FWVer;
}

/**********************************************************
Description: Module reset
Parameters:         
Return:      
Others:      
**********************************************************/
void BM32S4021_1::reset()
{
    uint8_t sendBuf[3] = {0x55, 0x10, 0x65};
    writeBytes(sendBuf,3);
    delay(25);
}

/**********************************************************
Description: Gets the module working mode
Parameters:    
Return:      mode: working mode
               0x00:Passive sending mode
               0x01:Active sending mode 
Others:      
**********************************************************/
uint8_t BM32S4021_1::getMode()
{
    uint8_t sendBuf[5] = {0x55, 0x80, 0x01, 0x01,0xD7};
    uint8_t buff[6] = {0};
    uint8_t mode = 0;
    writeBytes(sendBuf,5);
   if(readBytes(buff,6)== CHECK_OK)
   {
    mode= buff[4];
   }  
   delay(20);
   return mode;
}

/**********************************************************
Description: Get proximity sensing to shake the number
Parameters:    
Return:     debounce: Approximate sensing to shake off the number of times  
Others:      
**********************************************************/
uint8_t BM32S4021_1::getIRDebounce()
{
    uint8_t sendBuf[5] = {0x55, 0x80, 0x06, 0x01, 0xDC};
    uint8_t buff[6] = {0};
    uint8_t  debounce = 0;
    writeBytes(sendBuf,5);
    if(readBytes(buff,6)== CHECK_OK)
    {
     debounce= buff[4];
    }
    delay(20);
    return debounce;
}

/**********************************************************
Description: Get gesture trigger threshold
Parameters: 
Return:      threshold：gesture trigger threshold
Others:      
**********************************************************/
uint8_t BM32S4021_1::getGestureThreshold()
{
    uint8_t sendBuf[5] = {0x55, 0x80, 0x07, 0x01, 0xDD};
    uint8_t buff[6] = {0};
    uint8_t threshold = 0;
    writeBytes(sendBuf,5);
    if(readBytes(buff,6)== CHECK_OK)
    {
     threshold = buff[4];
    }
    delay(20);
    return threshold;
}

/**********************************************************
Description: Get IRQ Triger Time
Parameters: 
Return:      threshold: Close to the induction trigger threshold
Others:      
**********************************************************/
uint8_t BM32S4021_1::getIRQTrigerTime()
{
    uint8_t sendBuf[5] = {0x55, 0x80, 0x08, 0x01, 0xDE};
    uint8_t buff[6] = {0};
    uint8_t Time = 0;
    writeBytes(sendBuf,5);
    if(readBytes(buff,6)== CHECK_OK)
    {
     Time = buff[4];
    }
    delay(20);
    return Time;
}

/**********************************************************
Description: Get Cumulative continuous sliding duration
Parameters:  
Return:      Time: Cumulative continuous sliding duration parameter
Others:      Cumulative continuous sliding duration = Time×64ms
**********************************************************/
uint8_t BM32S4021_1::getIRContinutyGestureTime()
{
    uint8_t sendBuf[5] = {0x55, 0x80, 0x09, 0x01, 0xDF};
    uint8_t buff[6] = {0};
    uint8_t Time = 0;
    writeBytes(sendBuf,5);
    if(readBytes(buff,6)== CHECK_OK)
    {
     Time = buff[4];
    }
    delay(20);
    return Time;
}

/**********************************************************
Description: Get fastest gesture time
Parameters:  
Return:      Time: fastest gesture time parameter 
Others:      fastest gesture time parameter = (Time×5)+20 ms
**********************************************************/
uint8_t BM32S4021_1::getIRFastestGestureTime()
{
    uint8_t sendBuf[5] = {0x55, 0x80, 0x0A, 0x01, 0xE0};
    uint8_t buff[6] = {0};
    uint8_t Time = 0;
    writeBytes(sendBuf,5);
    if(readBytes(buff,6)== CHECK_OK)
    {
     Time = buff[4];
    }
    delay(20);
    return Time;
}

/**********************************************************
Description: Get slowest gesture time
Parameters:        
Return:      Time: The slowest gesture time parameter
Others:      slowest gesture time = Time×80ms
**********************************************************/
uint8_t BM32S4021_1::getIRSlowestGestureTime()
{
    uint8_t sendBuf[5] = {0x55, 0x80, 0x0B, 0x01, 0xE1};
    uint8_t buff[6] = {0};
    uint8_t Time = 0;
    writeBytes(sendBuf,5);
    if(readBytes(buff,6)== CHECK_OK)
    {
     Time = buff[4];
    }
    delay(20);
    return Time;
}

/**********************************************************
Description:  Get gesture hover time
Parameters:        
Return:       Time: gesture hover time
Others:       proximity sensing trigger time = Time×100ms
**********************************************************/
uint8_t BM32S4021_1::getGestureHoverTime()
{
    uint8_t sendBuf[5] = {0x55, 0x80, 0x0C, 0x01, 0xE2};
    uint8_t buff[6] = {0};
    uint8_t Time = 0;
    writeBytes(sendBuf,5);
    if(readBytes(buff,6)== CHECK_OK)
    {
     Time = buff[4];
    }
    delay(20);
    return Time;
}

/**********************************************************
Description: Set the module working mode
Parameters:  mode: working mode
               0x00:Passive sending mode
               0x01:Active sending mode 
Return:      Instruction execution
               0x00:Instruction executed successfully 
               0x01:Instruction execution failure  
Others:      
**********************************************************/
uint8_t BM32S4021_1::setMode(uint8_t mode)
{
    uint8_t sendBuf[6] = {0x55, 0xC0, 0x01, 0x01, 0x00, 0x00};
    sendBuf[4] =  mode;
    sendBuf[5] =  checksum(5,sendBuf);
    uint8_t buff[3] = {0};
    writeBytes(sendBuf,6);
    if(readBytes(buff,3)== CHECK_OK)
    {
     if(buff[1]== 0x7f)
     {
       delay(20);
       return 0x00;
     }
    }
     delay(20);
     return 0x01;
}

/**********************************************************
Description: Set approximate sensing to shake off the number of times
Parameters:  debounce: Approximate sensing to shake off the number of times, Radius(0~255)
Return:      Instruction execution
               0x00:Instruction executed successfully 
               0x01:Instruction execution failure 
Others:      
**********************************************************/
uint8_t BM32S4021_1::setIRDebounce(uint8_t  debounce)
{
    uint8_t sendBuf[6] = {0x55, 0xC0, 0x06, 0x01, 0x00, 0x00};
    sendBuf[4] =  debounce;
    sendBuf[5] =  checksum(5,sendBuf);
    uint8_t buff[3] = {0};
    writeBytes(sendBuf,6);
    if(readBytes(buff,3)== CHECK_OK)
    {
     if(buff[1]== 0x7f)
     {
       delay(20);
       return 0x00;
     }
    }
     delay(20);
     return 0x01;
}

/**********************************************************
Description: Set gesture trigger threshold
Parameters:  threshold:gesture trigger threshold, Radius(10~255)        
Return:      Instruction execution
               0x00:Instruction executed successfully 
               0x01:Instruction execution failure  
Others:       
**********************************************************/
uint8_t BM32S4021_1::setGestureThreshold(uint8_t  threshold)
{
    uint8_t sendBuf[6] = {0x55, 0xC0, 0x07, 0x01, 0x00, 0x00};
    sendBuf[4] =  threshold;
    sendBuf[5] =  checksum(5,sendBuf);
    uint8_t buff[3] = {0};
    writeBytes(sendBuf,6);
    if(readBytes(buff,3)== CHECK_OK)
    {
     if(buff[1]== 0x7f)
     {
       delay(20);
       return 0x00;
     }
    }
     delay(20);
     return 0x01;
}

/**********************************************************
Description: Set IRQ triger time
Parameters:  irTime:IRQ triger time parameter, Radius(0~255)       
Return:      Instruction execution
               0x00:Instruction executed successfully 
               0x01:Instruction execution failure   
Others:      IRQ triger time = (irTime×4)ms
**********************************************************/
uint8_t BM32S4021_1::setIRQTrigerTime(uint8_t  irTime)
{
    uint8_t sendBuf[6] = {0x55, 0xC0, 0x08, 0x01, 0x00, 0x00};
    sendBuf[4] =  irTime;
    sendBuf[5] =  checksum(5,sendBuf);
    uint8_t buff[3] = {0};
    writeBytes(sendBuf,6);
    if(readBytes(buff,3)== CHECK_OK)
    {
     if(buff[1]== 0x7f)
     {
       delay(20);
       return 0x00;
     }
    }
     delay(20);
     return 0x01;
}

/**********************************************************
Description: Set Cumulative continuous sliding duration
Parameters:  irTime: Cumulative continuous sliding duration parameter, Radius(0~255)        
Return:      Instruction execution
               0x00:Instruction executed successfully 
               0x01:Instruction execution failure  
Others:      Cumulative continuous sliding duration = irTime×64ms
**********************************************************/
uint8_t BM32S4021_1::setIRContinutyGestureTime(uint8_t  irTime)
{
    uint8_t sendBuf[6] = {0x55, 0xC0, 0x09, 0x01, 0x00, 0x00};
    sendBuf[4] =  irTime;
    sendBuf[5] =  checksum(5,sendBuf);
    uint8_t buff[3] = {0};
    writeBytes(sendBuf,6);
    if(readBytes(buff,3)== CHECK_OK)
    {
     if(buff[1]== 0x7f)
     {
       delay(20);
       return 0x00;
     }
    }
     delay(20);
     return 0x01;
}

/**********************************************************
Description: Set Fastest gesture judgment time
Parameters:  irTime: The fastest gesture judgment duration parameter, Radius(0~200)        
Return:      Instruction execution
               0x00:Instruction executed successfully 
               0x01:Instruction execution failure  
Others:      Fastest gesture judgment time = (irTime×4)+20 ms
**********************************************************/
uint8_t BM32S4021_1::setIRFastestGestureTime(uint8_t  irTime)
{
    uint8_t sendBuf[6] = {0x55, 0xC0, 0x0A, 0x01, 0x00, 0x00};
    sendBuf[4] =  irTime;
    sendBuf[5] =  checksum(5,sendBuf);
    uint8_t buff[3] = {0};
    writeBytes(sendBuf,6);
    if(readBytes(buff,3)== CHECK_OK)
    {
     if(buff[1]== 0x7f)
     {
       delay(20);
       return 0x00;
     }
    }
     delay(20);
     return 0x01;
}

/**********************************************************
Description: Set Slowest gesture judgment time
Parameters:  irTime: The Slowest gesture judgment duration parameter, Radius(0~200)        
Return:      Instruction execution
               0x00:Instruction executed successfully 
               0x01:Instruction execution failure  
Others:      Slowest gesture judgment time = (irTime×25) ms
**********************************************************/
uint8_t BM32S4021_1::setIRSlowestGestureTime(uint8_t  irTime)
{
    uint8_t sendBuf[6] = {0x55, 0xC0, 0x0B, 0x01, 0x00, 0x00};
    sendBuf[4] =  irTime;
    sendBuf[5] =  checksum(5,sendBuf);
    uint8_t buff[3] = {0};
    writeBytes(sendBuf,6);
    if(readBytes(buff,3)== CHECK_OK)
    {
     if(buff[1]== 0x7f)
     {
       delay(20);
       return 0x00;
     }
    }
     delay(20);
     return 0x01;
}


/**********************************************************
Description: Read IR1 reference
Parameters: 
Return:      ref: IR1 reference   
Others:      
**********************************************************/
uint8_t BM32S4021_1::readIR1Ref()
{
    uint8_t sendBuf[5] = {0x55, 0x80, 0x04, 0x01, 0xDA};
    uint8_t buff[6] = {0};
    uint8_t ref = 0;
    writeBytes(sendBuf,5);
    if(readBytes(buff,6)== CHECK_OK)
    {
     ref= buff[4];
    }
    delay(20);
    return ref;
}
/**********************************************************
Description: Read IR2 reference
Parameters: 
Return:      ref: IR2 reference   
Others:     
**********************************************************/
uint8_t BM32S4021_1::readIR2Ref()
{
    uint8_t sendBuf[5] = {0x55, 0x80, 0x05, 0x01, 0xDB};
    uint8_t buff[6] = {0};
    uint8_t ref = 0;
    writeBytes(sendBuf,5);
    if(readBytes(buff,6)== CHECK_OK)
    {
     ref= buff[4];
    }
    delay(20);
    return ref;
}

/**********************************************************
Description: Read IRStatus, IRGerstureNum, IR1Ref, IR2Ref value
Parameters:  buff[]: Stores the IR partial state parameter(4 bytes)
                 buff[0] : Gesture state 
                 buff[1] : number of left & right sliding
                 buff[2] : IR1 Ref 
                 buff[3] : IR2 Ref                         
Return:      Instruction execution
               0x00:Instruction executed successfully 
               0x01:Instruction execution failure    
Others:      
**********************************************************/
uint8_t BM32S4021_1::readIrA2_A5(uint8_t  buff[])
{
    uint8_t sendBuf[5] = {0x55, 0x80, 0x00, 0x04, 0xD9};
    uint8_t rbuf[9] ={0};
    writeBytes(sendBuf,5);
    if(readBytes(rbuf,9)== CHECK_OK)
    {
     delay(20);
     buff[0] = rbuf[4];
     buff[1] = rbuf[5];
     buff[2] = rbuf[6];
     buff[3] = rbuf[7];
     return 0x00;
    }
    delay(20);
    return 0x01;
}


/**********************************************************
Description: Get IR1 emission current value
Parameters:       
Return:      current: IR1 emission current value
Others:      
**********************************************************/
uint8_t BM32S4021_1::getIR1Current()
{
    uint8_t sendBuf[5] = {0x55, 0x80, 0x23, 0x01, 0xF9};
    uint8_t buff[6] = {0};
    uint8_t  current = 0;
    writeBytes(sendBuf,5);
    if(readBytes(buff,6)== CHECK_OK)
    {
     current = buff[4];
    }
    delay(20);
    return current;
}

/**********************************************************
Description: Get IR2 emission current value
Parameters:       
Return:      current: IR2 emission current value
Others:       
**********************************************************/
uint8_t BM32S4021_1::getIR2Current()
{
    uint8_t sendBuf[5] = {0x55, 0x80, 0x24, 0x01, 0xFA};
    uint8_t buff[6] = {0};
    uint8_t  current = 0;
    writeBytes(sendBuf,5);
    if(readBytes(buff,6)== CHECK_OK)
    {
     current = buff[4];
    }
    delay(20);
    return current;
}

/**********************************************************
Description: Set IR1 emission current value
Parameters:  current: IR1 emission current value parameter, Radius(0~31)        
Return:      Instruction execution
               0x00:Instruction executed successfully 
               0x01:Instruction execution failure  
Others:      R1 emission current value
                current≤15： 1+(1×current)mA
                current＞15： 16+1+1×(current-16)mA
**********************************************************/
uint8_t BM32S4021_1::setIR1Current(uint8_t  current)
{
    uint8_t sendBuf[6] = {0x55, 0xC0, 0x23, 0x01, 0x00, 0x00};
    sendBuf[4] =  current;
    sendBuf[5] =  checksum(5,sendBuf);
    uint8_t buff[3] = {0};
    writeBytes(sendBuf,6);
    if(readBytes(buff,3)== CHECK_OK)
    {
     if(buff[1]== 0x7f)
     {
       delay(20);
       return 0x00;
     }
    }
     delay(20);
     return 0x01;
}

/**********************************************************
Description: Set IR2 emission current value
Parameters:  current: IR2 emission current value parameter, Radius(0~31)        
Return:      Instruction execution
               0x00:Instruction executed successfully 
               0x01:Instruction execution failure  
Others:      R2 emission current value
                current≤15： 1+(1×current)mA
                current＞15： 16+1+1×(current-16)mA
**********************************************************/
uint8_t BM32S4021_1::setIR2Current(uint8_t  current)
{
    uint8_t sendBuf[6] = {0x55, 0xC0, 0x24, 0x01, 0x00, 0x00};
    sendBuf[4] =  current;
    sendBuf[5] =  checksum(5,sendBuf);
    uint8_t buff[3] = {0};
    writeBytes(sendBuf,6);
    if(readBytes(buff,3)== CHECK_OK)
    {
     if(buff[1]== 0x7f)
     {
       delay(20);
       return 0x00;
     }
    }
     delay(20);
     return 0x01;
}

/**********************************************************
Description: checksum
Parameters:  
             len: Data length
             data[]: Data 
Return:      result: checksum
Others:            
**********************************************************/
uint8_t BM32S4021_1::checksum(uint8_t len,uint8_t data[])
{
  uint8_t a=0;
  uint8_t result=0;
  for(a=0;a<len;a++)
   result  += data[a];
  return result;
}


/**********************************************************
Description: writeBytes
Parameters:  wbuf[]:Variables for storing Data to be sent
             wlen:Length of data sent  
Return:      void   
Others:
**********************************************************/
void BM32S4021_1::writeBytes(uint8_t wbuf[], uint8_t wlen)
{
  /* Select SoftwareSerial Interface */
  if (_softSerial != NULL)
  {
    while (_softSerial->available() > 0)
    {
      _softSerial->read();
    }
    _softSerial->write(wbuf, wlen);
  }
  /* Select HardwareSerial Interface */
  else
  {
    while (_hardSerial->available() > 0)
    {
      _hardSerial->read();
    }
    _hardSerial->write(wbuf, wlen);
  }
}

/**********************************************************
Description: readBytes
Parameters:  rbuf[]:Variables for storing Data to be obtained
             rlen:Length of data to be obtained
Return:      condition:
                      TIMEOUT_ERROR: Overtime
                      CHECK_OK: Check success
                      CHECK_ERROR: Check failure
Others:
**********************************************************/
uint8_t BM32S4021_1::readBytes(uint8_t rbuf[], uint8_t rlen, uint16_t timeOut)
{
  uint8_t i = 0, delayCnt = 0, checkSum = 0;
/* Select SoftwareSerial Interface */
  if (_softSerial != NULL)
  {
    for (i = 0; i < rlen; i++)
    {
      delayCnt = 0;
      while (_softSerial->available() == 0)
      {
        if (delayCnt > timeOut)
        {
          return TIMEOUT_ERROR; // Timeout error
        }
        delay(1);
        delayCnt++;
      }
      rbuf[i] = _softSerial->read();
    }
  }
/* Select HardwareSerial Interface */
  else
  {
    for (i = 0; i < rlen; i++)
    {
      delayCnt = 0;
      while (_hardSerial->available() == 0)
      {
        if (delayCnt > timeOut)
        {
          return TIMEOUT_ERROR; // Timeout error
        }
        delay(1);
        delayCnt++;
      }
      rbuf[i] = _hardSerial->read();
    }
  }

  /* check Sum */
  for (i = 0; i < (rlen - 1); i++)
  {
    checkSum += rbuf[i];
  }
  if (checkSum == rbuf[rlen - 1])
  {
    return CHECK_OK; // Check correct
  }
  else
  {
    return CHECK_ERROR; // Check error
  }
}
