#include "BM_IIC.h"
/*************************************************
Description:         IIC開始函數，發送一個開始信號。
Input:               無
Output:              無
Return:              uint8_t 0 scl被拉住 1正常 
Others:              其它說明
*************************************************/
uint8_t BM_SW_IIC::iic_start(void)
{
  pinMode(_scl, INPUT_PULLUP);
  pinMode(_sda, OUTPUT);
  digitalWrite(_sda, HIGH);
  if (digitalRead(_scl) == 0)
  {
    delayMicroseconds(10);
    if (digitalRead(_scl) == 0)
    {
      return 0;
    }
  }
  digitalWrite(_sda, LOW);
  pinMode(_scl, OUTPUT);
  digitalWrite(_scl, LOW);
  return 1;
}

/*************************************************
Description:         IIC停止函數，發送一個停止信號。
Input:               無
Output:              無
Return:              uint8_t 0 錯誤 1正常
Others:              其它說明
*************************************************/
uint8_t BM_SW_IIC::iic_stop(void)
{
  pinMode(_scl, INPUT_PULLUP);
  pinMode(_sda, OUTPUT);
  digitalWrite(_sda, HIGH);
  _time1 = 0;
  return 1;
}

/*************************************************
Description:         IIC等待ACK函數。
Input:               無
Output:              無
Return:              uint8_t   0  等到ACK
                               1  沒有等到ACK信號超時退出
                               2  SCL線被拉低
Others:              其它說明
*************************************************/
uint8_t BM_SW_IIC::iic_waitack(void)
{
  uint8_t i = 0;
  pinMode(_sda, INPUT_PULLUP);
  delayMicroseconds(1);
  pinMode(_scl, INPUT_PULLUP);

  if (digitalRead(_scl) == 0)
  {
    delayMicroseconds(10);
    if (digitalRead(_scl) == 0)
    {
      return 2;
    }
  }
  for (i = 0; i < 20; i++)
    if (digitalRead(_sda) == 0)
    {      
      pinMode(_sda, INPUT);
      pinMode(_scl, OUTPUT);
      digitalWrite(_scl, LOW);
      return 0;
    }
  pinMode(_scl, OUTPUT);
  digitalWrite(_scl, LOW);
  return 1;
}

/*************************************************
Description:         IIC發送ACK函數。
Input:               無
Output:              無
Return:              uint8_t 0 錯誤 1正常
Others:              其它說明
*************************************************/
uint8_t BM_SW_IIC::iic_ack(void)
{
  pinMode(_sda, OUTPUT);
  digitalWrite(_sda, LOW);
  pinMode(_scl, INPUT_PULLUP);
  delayMicroseconds(2);
  pinMode(_scl, OUTPUT);
  digitalWrite(_scl, LOW);
  return 1;
}

/*************************************************
Description:         IIC發送NACK函數。
Input:               無
Output:              無
Return:              uint8_t 0 錯誤 1正常
Others:              其它說明
*************************************************/
uint8_t BM_SW_IIC::iic_nack(void)
{
  pinMode(_sda, OUTPUT);
  digitalWrite(_sda, HIGH);
  pinMode(_scl, INPUT_PULLUP);
  delayMicroseconds(2);
  pinMode(_scl, OUTPUT);
  digitalWrite(_scl, LOW);
  digitalWrite(_sda, LOW);
  return 1;
}

/*************************************************
Description:         IIC發送一個字節函數。
Input:               @param data 發送的數據
Output:              無
Return:              uint8_t 0 scl線被拉住 1成功
Others:              其它說明
*************************************************/
uint8_t BM_SW_IIC::iic_write(uint8_t data)
{
  pinMode (_sda, OUTPUT);
  uint8_t i = 0;
  uint8_t y = 0;
  for (i = 0; i < 8; i++)
  {
    delay();
    if ((data & 0x80) == 0x80)
      digitalWrite(_sda, HIGH);
    else
      digitalWrite(_sda, LOW);
    pinMode(_scl, INPUT_PULLUP);
    if (digitalRead(_scl) == 0)
    {
      delayMicroseconds(10);
      if (digitalRead(_scl) == 0)
        return 0;
    }
    pinMode(_scl, OUTPUT);
    digitalWrite(_scl, LOW);
    data = data << 1;
  }
  return 1;
}

/*************************************************
Description:         IIC讀取一個字節函數
Input:               @pagram uint8_t* flag 0scl線被拉住 1 成功
Output:              無
Return:              uint8_t data 讀取的數據
Others:              其它說明
*************************************************/
uint8_t BM_SW_IIC::iic_read(uint8_t* flag)
{
  pinMode (_sda,INPUT_PULLUP);
  uint8_t data = 0;
  uint8_t i = 0;
  uint8_t y = 0;
  *flag = 1;
  for (i = 0; i < 8; i++)
  {
    delay();
    pinMode(_scl, INPUT_PULLUP);

    if (digitalRead(_scl) == 0)
    {
      delayMicroseconds(10);
      if (digitalRead(_scl) == 0)
        *flag = 0;
      //  return 0;
    }
    data = data << 1;
    if (digitalRead(_sda) == 1)
      data = data + 1;
    pinMode(_scl, OUTPUT);
    digitalWrite(_scl, LOW);
  }
  return data;
}

/*************************************************
Description:         IIC初始化函數，初始化函數。
Input:               無
Output:              無
Return:              無
Others:              其它說明
*************************************************/
void BM_SW_IIC::begin(void)
{
  _scl = 14;
  _sda = 15;

  pinMode (_scl, OUTPUT);
  pinMode (_sda, OUTPUT);

  digitalWrite(_sda, HIGH);
  digitalWrite(_scl, HIGH);
  _iicClock = 0;

  _timeEn = 0;;
  _timeOut = 0;
  _timeFlag = 0;
  _time1 = 0;
  _time2 = 0;
  _begin = 1;
}

void BM_SW_IIC::end(void)
{
  _begin = 0;
}
/*************************************************
Description:         延時_timer個時鐘週期
Input:               無
Output:              無
Return:              無
Others:              其它說明
*************************************************/
void BM_SW_IIC::delay()
{
  uint8_t i = 0;
  uint8_t t = 0;
  for (i = 0; i < _iicClock; i++)
    t++;
}

void BM_SW_IIC::setClock(uint32_t clockDiv)
{
  _iicClock = clockDiv;
}
/*************************************************
Description:         IIC地址匹配函數，發送開始信號、地址數據、等待ACK。
Input:               無
Output:              無
Return:              無 
Others:              其它說明
*************************************************/
void BM_SW_IIC::beginTransmission(uint8_t addr)
{
  if(_begin == 0)
    return ;
  write(addr << 1);
  return ;
}

/*************************************************
Description:         IIC發送哦一個字節，且等待ACK。
Input:               @param data 發送的數據
Output:              無
Return:              返回        1   成功
                                 0  失敗
Others:              其它說明
*************************************************/
uint8_t BM_SW_IIC::write(uint8_t data)
{
  if(_begin == 0)
    return 0;
  _txBuf[_txBufNum++] = data;
 return 1;
}

/*************************************************
Description:         IIC結束數據傳輸
Input:               @param stop   0  發送完數據不停止時不發送STOP信號
                                   1  發送完數據停止時發送STOP信號
Output:              無
Return:              uint8_t        0 正常
                                    1 數據超出緩存最大值
                                    2 無回應ACK
                                    4 其他原因
                                    5 超時
Others:              其它說明
*************************************************/
uint8_t BM_SW_IIC::endTransmission(uint8_t stop)
{
  uint8_t i = 0;
  uint8_t ack_flag = 0;
  if(_begin == 0)                 //判斷是否開始傳輸
    return 4;
  unsigned long data = 0;
  if(_txBufNum > 255)             //判斷發送的數據是否超出數據緩存大小
  {
    _txBufNum = 0;
    return 1;
  }
  _time1 = micros();
  if(iic_start() == 0)
    return 4;
    
  iic_write(_txBuf[i++]);
  ack_flag = iic_waitack();
  if(ack_flag != 0)
  {
    iic_stop();
    _txBufNum = 0;
    if(ack_flag == 1)
      return 2;
    else 
      return 4;
  }
  for(;i < _txBufNum;i++)
  {
    if(_timeEn == 1)
    {
      _time2 = micros();
      if(_time2 > _time1)
      {
        if((_time2 - _time1) > _timeOut)
        {
          iic_stop();
          _txBufNum = 0;
          _timeFlag = 1;
          return 5;
        }
      }
      else
      {
        data = _time1 - _time2;
        data = 4294967295-data + 1;
        if(data > _timeOut)
        {
          iic_stop();
          _txBufNum = 0;
          _timeFlag = 1;
          return 5;
        }
      }  
    }
    iic_write(_txBuf[i]);
    ack_flag = iic_waitack();
    if(ack_flag != 0)
    {
      iic_stop();
      _txBufNum = 0;
      if(ack_flag == 1)
        return 2;
      else 
      return 4;
    }
  }
  
  if (stop == 1)
    iic_stop();
  _txBufNum = 0;
  return 0;
}

/*************************************************
Description:         讀取len個字節數據到addr中
Input:               @param addr 讀取的數據數組
                     @param len 讀取的數據字節數
                     @param sendStop 讀取完數據後是否發送stop信號  0 不發送 1 發送
Output:              無
Return:              uint8_t data 讀取的數據
Others:              其它說明
*************************************************/
uint8_t BM_SW_IIC::requestFrom(uint8_t addr, uint8_t len, uint8_t sendStop)
{
  if(_begin == 0)
    return 4;
  uint8_t i = 0;
  unsigned long data = 0;
  uint8_t flag = 0;
  _time1 = micros();
  iic_start();
  iic_write((addr << 1) + 1);
  if(iic_waitack() != 0)
  {
    _rxBufNum = 0;
    iic_stop();
  //  return 2;
  }
  for (i = 0; i < len - 1; i++)
  {
    if(_timeEn == 1)
    {
      _time2 = micros();
      if(_time2 > _time1)
      {
        if((_time2 - _time1) > _timeOut)
        {
          iic_stop();
          _rxBufNum = 0;
          _timeFlag = 1;
          
       //   return 5;
        }
      }
      else
      {
        data = _time1 - _time2;
        data = 4294967295-data + 1;
        if(data > _timeOut)
        {
          iic_stop();
          _rxBufNum = 0;
          _timeFlag = 1;
       //   return 5;
        }
      } 
    }
    _rxBuf[i] = iic_read(&flag);
    iic_ack();
  }
  _rxBuf[i] = iic_read(&flag);
  iic_nack();
  _rxBufNum = len;
  _bufReadNum = 0;
  if (sendStop == 1)
    iic_stop();
  return len;
}

/*************************************************
Description:         當前緩存中未讀取的數據的字節數
Input:               無
Output:              無
Return:              uint8_t 緩存中未讀取的數據的字節數
Others:              其它說明
*************************************************/
uint8_t BM_SW_IIC::available(void)
{
  return _rxBufNum - _bufReadNum;
}

/*************************************************
Description:         從緩存中讀取一個字節數據
Input:               無
Output:              無
Return:              uint8_t data 讀取的數據
Others:              其它說明
*************************************************/
uint8_t BM_SW_IIC::read(void)
{
  uint8_t i = 0;
  
  if ((_rxBufNum - _bufReadNum) == 0)
    return -1;
  i = _rxBuf[_bufReadNum];
  _bufReadNum++;
  return i;

}

/*************************************************
Description:         從緩存中讀取一個字節數據,並且不刪除該數據緩存
Input:               無
Output:              無
Return:              uint8_t data 讀取的數據
Others:              其它說明
*************************************************/
uint8_t BM_SW_IIC::peek(void)
{
  uint8_t i = 0;
  
  if ((_rxBufNum - _bufReadNum) == 0)
    return -1;
  i = _rxBuf[_bufReadNum];
  _bufReadNum;
  return i;

}

/*************************************************
Description:         設置超時參數
Input:               @param timeout 超時時間
                     @param reset_with_timeout 超時開關 0 關閉 1 開啟
Output:              無
Return:              無
Others:              其它說明
*************************************************/
void BM_SW_IIC::setWireTimeout(uint32_t timeout,uint8_t reset_with_timeout)
{
  _timeEn = reset_with_timeout;
  _timeOut = timeout;
}

/*************************************************
Description:         獲取超時標誌
Input:               無 
Output:              無
Return:              uint8_t data 讀取的數據
Others:              uint8_t _timeFlage 超時標誌
*************************************************/
bool BM_SW_IIC::getWireTimeoutFlag(void)
{
  if(_timeFlag == 0)
    return false;
  else
    return true;
}

/*************************************************
Description:         清除超時標誌
Input:               無 
Output:              無
Return:              無
Others:              uint8_t _timeFlage 超時標誌
*************************************************/
uint8_t BM_SW_IIC::clearWireTimeoutFlag(void)
{
  _timeFlag = 0;
  return 0;
}

/*************************************************
Description:         改變sck、sda對應引腳
Input:               @param iic_sck sck對應引腳
                     @param iic_sda sda對應引腳
Output:              無
Return:              無
Others:              其它說明
*************************************************/
void BM_SW_IIC::ChangeIICPin(uint8_t iic_sck, uint8_t iic_sda)
{
  _scl = iic_sck;
  _sda = iic_sda;

  pinMode (_sda, OUTPUT);
  pinMode (_scl, OUTPUT);

  digitalWrite(_sda, HIGH);
  digitalWrite(_scl, HIGH);
}

/*************************************************
Description:         BM_IIC的構造函數，設置scl、sda引腳，使用該構造函數創建對象IIC為模擬IIC。
Input:               @param scl scl引腳
                     @param sda sda引腳
Output:              無
Return:              無
Others:              其它說明
*************************************************/
BM_IIC::BM_IIC(uint8_t scl,uint8_t sda)
{
    _the_BM_IIC = &_BM_IIC;
  _the_TwoWire = NULL;
  _addr = 0x00;
  _scl = scl;
  _sda = sda;
}

/*************************************************
Description:         BM_IIC的構造函數，設置從機地址，scl、sda引腳，使用該構造函數創建對象IIC為模擬IIC。
Input:               @param addr 從機地址
                     @param scl scl引腳
                     @param sda sda引腳
Output:              無
Return:              無
Others:              其它說明
*************************************************/
BM_IIC::BM_IIC(uint8_t addr, uint8_t scl, uint8_t sda)
{
  _the_BM_IIC = &_BM_IIC;
  _the_TwoWire = NULL;
  _addr = addr;
  _scl = scl;
  _sda = sda;
}

/*************************************************
Description:         BM_IIC的構造函數，使用該構造函數創建對象IIC為硬件IIC。
Input:               無
Output:              無
Return:              無
Others:              其它說明
*************************************************/
BM_IIC::BM_IIC()
{
  _the_BM_IIC = NULL;
  _the_TwoWire = &Wire;
  _addr = 0x00;
}
/*************************************************
Description:         BM_IIC的初始化函數。
Input:               @param addr 從機地址
Output:              無
Return:              無
Others:              其它說明
*************************************************/
BM_IIC::BM_IIC(uint8_t addr)
{
  _the_BM_IIC = NULL;
  _the_TwoWire = &Wire;
  _addr = addr;
}

/*************************************************
Description:         BM_IIC的初始化函數。
Input:               無
Output:              無
Return:              無
Others:              其它說明
*************************************************/
void BM_IIC::begin()
{
  if (_the_BM_IIC != NULL)
  {
    _the_BM_IIC->begin();
    _the_BM_IIC->ChangeIICPin(_scl, _sda);
  }
  else if (_the_TwoWire != NULL)
  {
    _the_TwoWire->begin();
  }

}

/*************************************************
Description:         BM_IIC的初始化函數。
Input:               @param addr 從機地址
Output:              無
Return:              無
Others:              其它說明
*************************************************/
void BM_IIC::begin(uint8_t address)
{
  if (_the_TwoWire != NULL)
  {
    _the_TwoWire->begin(address);
  }

}

/*************************************************
Description:         BM_IIC的初始化函數。
Input:               @param addr 從機地址
Output:              無
Return:              無
Others:              其它說明
*************************************************/
void BM_IIC::begin(int address)
{
  if (_the_TwoWire != NULL)
  {
    _the_TwoWire->begin(address);
  }
}

/*************************************************
Description:         關閉IIC
Input:               無
Output:              無
Return:              無
Others:              其它說明
*************************************************/
void BM_IIC::end(void)
{
  if (_the_BM_IIC != NULL)
  {
    _the_BM_IIC->end();
  }
  else if (_the_TwoWire != NULL)
  {
    _the_TwoWire->end();
  }

}

/*************************************************
Description:         設置IIC通信速率
Input:               @param clockDiv 通信速率
Output:              無
Return:              無
Others:              其它說明
*************************************************/
void BM_IIC::setClock(uint32_t clockDiv)
{
  if (_the_BM_IIC != NULL)
  {
    _the_BM_IIC->setClock(clockDiv);
  }
  else if (_the_TwoWire != NULL)
  {
    _the_TwoWire->setClock(clockDiv);
  }
}

/*************************************************
Description:         設置超時參數
Input:               @param timeout 超時時間
                     @param reset_with_timeout 超時開關 0 關閉 1 開啟
Output:              無
Return:              無
Others:              其它說明
*************************************************/
void BM_IIC::setWireTimeout(uint32_t timeout,uint8_t reset_with_timeout)
{
  if (_the_BM_IIC != NULL)
  {
    _the_BM_IIC->setWireTimeout(timeout,reset_with_timeout);
  }
  else if (_the_TwoWire != NULL)
  {
    _the_TwoWire->setWireTimeout(timeout,reset_with_timeout);
  }
}

/*************************************************
Description:         獲取超時標誌
Input:               無
Output:              無
Return:              無
Others:              其它說明
*************************************************/
bool BM_IIC::getWireTimeoutFlag(void)
{
  if (_the_BM_IIC != NULL)
  {
    return _the_BM_IIC->getWireTimeoutFlag();
  }
  else if (_the_TwoWire != NULL)
  {
    return _the_TwoWire->getWireTimeoutFlag();
  }
}

/*************************************************
Description:         清除超時標誌
Input:               無
Output:              無
Return:              無
Others:              其它說明
*************************************************/
uint8_t BM_IIC::clearWireTimeoutFlag(void)
{
  if (_the_BM_IIC != NULL)
  {
    _the_BM_IIC->clearWireTimeoutFlag();
  }
  else if (_the_TwoWire != NULL)
  {
    _the_TwoWire->clearWireTimeoutFlag();
  }
}

/*************************************************
Description:         從機接收函數
Input:               @param fun 從機接收函數指針
Output:              無
Return:              無
Others:              其它說明
*************************************************/
void BM_IIC::onReceive(void (*handler)(int))
{
  if (_the_TwoWire != NULL)
  {
    _the_TwoWire->onReceive(handler);
  }
}

/*************************************************
Description:         從機發送函數
Input:               @param fun 從機發送函數指針
Output:              無
Return:              無
Others:              其它說明
*************************************************/
void BM_IIC::onRequest(void (*handler)(void))
{
  if (_the_TwoWire != NULL)
  {
    _the_TwoWire->onRequest(handler);
  }
}

/*************************************************
Description:         讀取len個字節數據到buffer數組中
Input:               @param buffer 讀取的數據數組
                     @param len 讀取的數據字節數
                     @param stop 讀取完數據是否發送停止位  0 不發送stop信號  1 發送stop信號
Output:              無
Return:              bool true讀取成功 false讀取失敗
Others:              其它說明
*************************************************/
uint8_t BM_IIC::readBytes(uint8_t *buffer, uint8_t len, uint8_t stop)
{
  uint8_t i = 0;
  if (_the_BM_IIC != NULL)
  {
    _the_BM_IIC->requestFrom(_addr, len, stop);
    while (_the_BM_IIC->available() != 0)
      buffer[i++] =  _the_BM_IIC->read();
    return i;
  }
  else if (_the_TwoWire != NULL)
  {
    _the_TwoWire->requestFrom(_addr, len, stop);
    while (_the_TwoWire->available() != 0)
      buffer[i++] =  _the_TwoWire->read();
    return i;
  }
  return false;
}

/*************************************************
Description:         讀取一個個字節數據
Input:               @param stop 讀取完數據是否發送停止位  0 不發送stop信號  1 發送stop信號
Output:              無
Return:              uint8_t 讀取的數據
Others:              其它說明
*************************************************/
uint8_t BM_IIC::readByte( uint8_t stop)
{
  uint8_t data = 0;
  uint8_t i = 0;
  if (_the_BM_IIC != NULL)
  {
    _the_BM_IIC->requestFrom(_addr, i, stop);
      return _the_BM_IIC->read();
  }
  else if (_the_TwoWire != NULL)
  {
    _the_TwoWire->requestFrom(_addr, i, stop);
      return _the_TwoWire->read();
  }
  return false;
}

/*************************************************
Description:         發送len個字節的數據從buffer數組
Input:               @param buffer 要發送的數據
                     @param len 要發送的字節數 
                     @param stop 發送完是否發送stop信號  0 不發送 1 發送
Output:              無
Return:              bool true 發送成功 false 發送失敗
Others:              其它說明
*************************************************/
uint8_t BM_IIC::writeBytes(uint8_t *buffer, uint8_t len, uint8_t stop)
{
  uint8_t i = 0;
  if (_the_BM_IIC != NULL)
  {
    _the_BM_IIC->beginTransmission(_addr);
    for (i = 0; i < len; i++)
      _the_BM_IIC->write(buffer[i]);
    return _the_BM_IIC->endTransmission(stop);
  }
  else if (_the_TwoWire != NULL)
  {
    _the_TwoWire->beginTransmission(_addr);
    for (i = 0; i < len; i++)
      _the_TwoWire->write(buffer[i]);
    return _the_TwoWire->endTransmission(stop);
  }
  return false;
}
/*************************************************
Description:         發送一個個字節數據
Input:               @param stop 讀取完數據是否發送停止位  0 不發送stop信號  1 發送stop信號
Output:              無
Return:              uint8_t 讀取的數據
Others:              其它說明
*************************************************/
uint8_t BM_IIC::writeByte(uint8_t data,uint8_t stop)
{
  if (_the_BM_IIC != NULL)
  {
    _the_BM_IIC->beginTransmission(_addr);
      _the_BM_IIC->write(data);
    return _the_BM_IIC->endTransmission(stop);

  }
  else if (_the_TwoWire != NULL)
  {
    _the_TwoWire->beginTransmission(_addr);
      _the_TwoWire->write(data);
    return _the_TwoWire->endTransmission(stop);

  }
  return false;
}

/*************************************************
Description:         IIC發送地址和等待ACK
Input:               @param addr 從機地址
Output:              無
Return:              無
Others:              其它說明
*************************************************/
void BM_IIC::beginTransmission(uint8_t addr)
{
  if (_the_BM_IIC != NULL)
  {
    _the_BM_IIC->beginTransmission(addr);
  }
  else if (_the_TwoWire != NULL)
  {
    _the_TwoWire->beginTransmission(addr);
  }
}

/*************************************************
Description:         IIC發送地址和等待ACK
Input:               @param addr 從機地址
Output:              無
Return:              無
Others:              其它說明
*************************************************/
void BM_IIC::beginTransmission(int addr)
{
  if (_the_BM_IIC != NULL)
  {
    _the_BM_IIC->beginTransmission((uint8_t)addr);
  }
  else if (_the_TwoWire != NULL)
  {
    _the_TwoWire->beginTransmission((uint8_t)addr);
  }
}

/*************************************************
Description:         停止發送數據
Input:               無
Output:              無
Return:              bool false 函數失敗 true 函數成功
Others:              其它說明
*************************************************/
uint8_t BM_IIC::endTransmission(void)
{
  if (_the_BM_IIC != NULL)
  {
    _the_BM_IIC->endTransmission(1);
    return true;
  }
  else if (_the_TwoWire != NULL)
  {
    _the_TwoWire->endTransmission(1);
    return true;
  }
  return false;
}
/*************************************************
Description:         停止發送數據
Input:               @param stop 1 發送stop信號 0 不發送stop信號
Output:              無
Return:              bool false 函數失敗 true 函數成功
Others:              其它說明
*************************************************/
uint8_t BM_IIC::endTransmission(uint8_t stop)
{
  if (_the_BM_IIC != NULL)
  {
    return _the_BM_IIC->endTransmission(stop);
  }
  else if (_the_TwoWire != NULL)
  {
    return _the_TwoWire->endTransmission(stop);
  }
  return false;
}

/*************************************************
Description:         寫入一個字節到發送緩存
Input:               @param data 寫入的數據
Output:              無
Return:              bool false 函數失敗 ture 函數成功
Others:              其它說明
*************************************************/
uint8_t BM_IIC::write(uint8_t data)
{
  if (_the_BM_IIC != NULL)
  {
    _the_BM_IIC->write(data);
    return true;
  }
  else if (_the_TwoWire != NULL)
  {
    _the_TwoWire->write(data);
    return true;
  }
  return false;
}

/*************************************************
Description:         寫入一個unsigned long型數據到發送緩存
Input:               @param data 寫入的數據
Output:              無
Return:              bool false 函數失敗 ture 函數成功
Others:              其它說明
*************************************************/
uint8_t BM_IIC::write(unsigned long data)
{
  if (_the_BM_IIC != NULL)
  {
    _the_BM_IIC->write((uint8_t)data);
    return true;
  }
  else if (_the_TwoWire != NULL)
  {
    _the_TwoWire->write((uint8_t)data);
    return true;
  }
}

/*************************************************
Description:         寫入一個long型數據到發送緩存
Input:               @param data 寫入的數據
Output:              無
Return:              bool false 函數失敗 ture 函數成功
Others:              其它說明
*************************************************/
uint8_t BM_IIC::write(long data)
{
  if (_the_BM_IIC != NULL)
  {
    _the_BM_IIC->write((uint8_t)data);
    return true;
  }
  else if (_the_TwoWire != NULL)
  {
    _the_TwoWire->write((uint8_t)data);
    return true;
  }
}

/*************************************************
Description:         寫入一個unsigned int型數據到發送緩存
Input:               @param data 寫入的數據
Output:              無
Return:              bool false 函數失敗 ture 函數成功
Others:              其它說明
*************************************************/
uint8_t BM_IIC::write(unsigned int data)
{
  if (_the_BM_IIC != NULL)
  {
    _the_BM_IIC->write((uint8_t)data);
    return true;
  }
  else if (_the_TwoWire != NULL)
  {
    _the_TwoWire->write((uint8_t)data);
    return true;
  }
}

/*************************************************
Description:         寫入一個int型數據到發送緩存
Input:               @param data 寫入的數據
Output:              無
Return:              bool false 函數失敗 ture 函數成功
Others:              其它說明
*************************************************/
uint8_t BM_IIC::write(int data)
{
  if (_the_BM_IIC != NULL)
  {
    _the_BM_IIC->write((uint8_t)data);
    return true;
  }
  else if (_the_TwoWire != NULL)
  {
    _the_TwoWire->write((uint8_t)data);
    return true;
  }
}

/*************************************************
Description:         IIC讀取緩存中一個字節數據，並且刪除在緩存中刪除該數據
Input:               無
Output:              無
Return:              uint8_t data 讀取的數據
Others:              其它說明
*************************************************/
uint8_t BM_IIC::read(void)
{
  if (_the_BM_IIC != NULL)
  {
    return _the_BM_IIC->read();
  }
  else if (_the_TwoWire != NULL)
  {
    return _the_TwoWire->read();
  }
}

/*************************************************
Description:         IIC讀取緩存中一個字節數據，並且不會刪除該數據的緩存
Input:               無
Output:              無
Return:              uint8_t data 讀取的數據
Others:              其它說明
*************************************************/
uint8_t BM_IIC::peek(void)
{
  if (_the_BM_IIC != NULL)
  {
    return _the_BM_IIC->peek();
  }
  else if (_the_TwoWire != NULL)
  {
    return _the_TwoWire->peek();
  }
}

/*************************************************
Description:         當前緩存中未讀取的數據的字節數
Input:               無
Output:              無
Return:              uint8_t 緩存中未讀取的數據的字節數
Others:              其它說明
*************************************************/
uint8_t BM_IIC::available(void)
{
  if (_the_BM_IIC != NULL)
  {
    return _the_BM_IIC->available();
  }
  else if (_the_TwoWire != NULL)
  {
    return _the_TwoWire->available();
  }
}

/*************************************************
Description:         讀取len個字節數據到addr中
Input:               @param addr 讀取的數據數組
                     @param len 讀取的數據字節數
                     @param sendStop 讀取完數據後是否發送stop信號  0 不發送 1 發送
Output:              無
Return:              uint8_t data 讀取的數據個數
Others:              其它說明
*************************************************/
uint8_t BM_IIC::requestFrom(uint8_t addr, uint8_t quantity, uint8_t sendStop)
{
  if (_the_BM_IIC != NULL)
  {
    return _the_BM_IIC->requestFrom(addr, quantity, sendStop);
  }
  else if (_the_TwoWire != NULL)
  {
    return _the_TwoWire->requestFrom(addr, quantity, sendStop);
  }
}

/*************************************************
Description:         讀取len個字節數據到addr中
Input:               @param addr 讀取的數據數組
                     @param len 讀取的數據字節數
Output:              無
Return:              uint8_t data 讀取的數據個數
Others:              其它說明
*************************************************/
uint8_t BM_IIC::requestFrom(uint8_t addr, uint8_t quantity)
{
  if (_the_BM_IIC != NULL)
  {
    return _the_BM_IIC->requestFrom((uint8_t )addr, (uint8_t )quantity,(uint8_t) 1);
  }
  else if (_the_TwoWire != NULL)
  {
    return _the_TwoWire->requestFrom((uint8_t )addr,(uint8_t ) quantity,(uint8_t)1);
  }
}

/*************************************************
Description:         讀取len個int型數據到addr中
Input:               @param addr 讀取的數據數組
                     @param len 讀取的數據字節數
                     @param sendStop 讀取完數據後是否發送stop信號  0 不發送 1 發送
Output:              無
Return:              uint8_t data 讀取的數據個數
Others:              其它說明
*************************************************/
uint8_t BM_IIC::requestFrom(int addr, int quantity,int sendStop)
{
  if (_the_BM_IIC != NULL)
  {
    return _the_BM_IIC->requestFrom((uint8_t )addr, (uint8_t )quantity, (uint8_t)sendStop);
  }
  else if (_the_TwoWire != NULL)
  {
    return _the_TwoWire->requestFrom((uint8_t)addr,(uint8_t)quantity,(uint8_t)sendStop);
  }
}  


/*************************************************
Description:         讀取len個int型數據到addr中
Input:               @param addr 讀取的數據數組
                     @param len 讀取的數據字節數
Output:              無
Return:              uint8_t data 讀取的數據個數
Others:              其它說明
*************************************************/
uint8_t BM_IIC::requestFrom(int addr, int quantity)
{
  if(_the_BM_IIC != NULL)
  {
    return _the_BM_IIC->requestFrom((uint8_t )addr, (uint8_t )quantity,(uint8_t) 1);
  }
  else if (_the_TwoWire != NULL)
  {
    return _the_TwoWire->requestFrom((uint8_t )addr,(uint8_t ) quantity,(uint8_t)1);
  }
}    

/*************************************************
Description:         讀取len個字節數據從從機的寄存器中
Input:               @param addr 讀取的數據數組
                     @param len 讀取的數據字節數
                     @param iaddress 從機寄存器地址
                     @param isize從機寄存器地址大小
                     @param sendStop 是否發送停止位
Output:              無
Return:              uint8_t data 讀取的數據個數
Others:              其它說明
*************************************************/
uint8_t BM_IIC::requestFrom(uint8_t addr, uint8_t quantity,uint32_t iaddress,uint8_t isize,uint8_t sendStop)
{
 uint8_t i = 0;
 if (_the_BM_IIC != NULL)
  {
    _the_BM_IIC->beginTransmission(addr);
    while (isize-- > 0)
    {
      _the_BM_IIC->write((uint8_t)(iaddress >> (isize*8)));
    }
     _the_BM_IIC->endTransmission(0);
    return _the_BM_IIC->requestFrom((uint8_t )addr, (uint8_t )quantity, sendStop);
  }
  else if (_the_TwoWire != NULL)
  {
    return _the_TwoWire->requestFrom(addr,quantity,iaddress,isize,sendStop);
  }
}

/*************************************************
Description:         返回緩存最大存儲個數
Input:               無
Output:              無
Return:              uint8_t 緩存最大存儲個數
Others:              其它說明
*************************************************/
uint8_t BM_IIC::maxBufferSize(void)
{
  if (_the_BM_IIC != NULL)
  {
    return 255;
  }
  else if (_the_TwoWire != NULL)
  {
    return 32;
  }
}
